<?php declare(strict_types=1);
namespace Newland\NeosViewHelpers\ViewHelpers\Media;

/*
 * This file is part of the "neos-viewhelpers" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\FluidAdaptor\Core\ViewHelper\AbstractViewHelper;
use Neos\Media\Domain\Model\Image;
use Neos\Utility\Files;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;
use function Safe\mime_content_type;
use function Safe\stream_get_contents;

/**
 * Inline view helper.
 *
 * Outputs an inline-version of the given media node.
 */
class InlineViewHelper extends AbstractViewHelper
{

    public function initializeArguments(): void
    {
        $this->registerArgument('image', Image::class,  'An image resource', false, null);
        $this->registerArgument('path', Image::class,  'A path to an image resource.', false, null);
        $this->registerArgument('class', Image::class,  'Additional class for the SVG tag if it is an SVG.', false, null);
    }

    public static function renderStatic(
        array $arguments,
        \Closure $renderChildrenClosure,
        RenderingContextInterface $renderingContext
    ): string {
        $contents = null;
        $mimeType = null;

        /** @var Image|null $image */
        $image = $arguments['image'];

        if ($image) {
            $contents = static::getFileContentsFromObject($image);
            $mimeType = $image->getMediaType();
        } elseif ($arguments['path']) {
            $contents = static::getFileContentsFromPath($arguments['path']);
            if ($contents) {
                $mimeType = mime_content_type(FLOW_PATH_ROOT . $arguments['path']);
            }
        }

        if ($contents === null || $mimeType === null) {
            return '';
        }

        if ($mimeType === 'image/svg+xml') {
            return static::prepareSvg($contents, $arguments['class']);
        }

        return static::prepareBlob($mimeType, $contents);
    }

    protected static function prepareBlob(string $mimeType, string $contents): string
    {
        return 'data:' . $mimeType . ';base64, ' . base64_encode($contents);
    }

    protected static function prepareSvg(string $contents, string $class = null): string
    {
        if ($class) {
            return static::addClassToSvgContents($contents, $class);
        }

        return $contents;
    }

    protected static function getFileContentsFromObject(Image $image): ?string
    {
        $stream = $image->getResource()->getStream();

        if (is_resource($stream)) {
            $contents = stream_get_contents($stream);
            fclose($stream);
            return (string) $contents;
        }

        return null;
    }

    private static function getFileContentsFromPath($path): ?string
    {
        if (file_exists(FLOW_PATH_ROOT . $path)) {
            return Files::getFileContents(FLOW_PATH_ROOT . $path);
        }
        return null;
    }

    private static function addClassToSvgContents(string $contents, string $class): string
    {
        return str_replace(
            '<svg',
            '<svg class="' . $class . '" ',
            $contents
        );
    }
}
