<?php
namespace Newland\SocialGrid\Post;

use GuzzleHttp\Client as GuzzleClient;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cache\CacheManager;

class OembedPost extends AbstractPost
{
    const CACHE_NAME = 'Newland_SocialGrid-Post';
    const CACHE_KEY = 'social-grid-post-{postId}';

    /** @var string */
    private $url;
    private $postId;

    /**
     * @Flow\Inject
     * @var CacheManager
     */
    protected $cacheManager;

    /**
     * @var GuzzleClient
     */
    protected $httpClient;

    public function __construct(string $type, string $url)
    {
        $this->httpClient = new GuzzleClient();
        $this->postId = $this->extractPostId($url);

        switch ($type) {
            case 'instagram':
                $this->url = 'https://api.instagram.com/oembed?url=' . $url;
                break;
            default:
                $this->url = 'https://www.facebook.com/plugins/post/oembed.json/?url='
                    . $url
                    . '&omitscript=true&maxwidth=750';
                break;
        }
    }

    private function getPostFromCacheOrApi()
    {
        $cache = $this->cacheManager->getCache(static::CACHE_NAME);
        $cacheKey = str_replace(static::CACHE_KEY, '{postId}', $this->postId);

        if ($cache->has($cacheKey)) {
            return $cache->get($cacheKey);
        }

        $information = $this->fetchPostFromApi();
        $cache->set($cacheKey, $information);
        return $information;
    }

    private function fetchPostFromApi()
    {
        $response = $this->httpClient->get($this->url);
        $result = json_decode($response->getBody(), false);

        if (!$result || $response->getStatusCode() !== 200) {
            return (object) [];
        }

        return $result;
    }

    private function extractPostId(string $url)
    {
        $parsedUrl = parse_url($url);
        if (!$parsedUrl) {
            throw new \Exception('The URL ' . $url . ' could not be parsed.');
        }

        $pathParts = explode('/', $parsedUrl['path']);
        return (string) array_pop($pathParts);
    }

    public function getPost()
    {
        return (object) $this->getPostFromCacheOrApi();
    }
}
