<?php declare(strict_types=1);
namespace Newland\NeosRoutingBehaviourHooks\Tests\Unit\Behaviour;

use GuzzleHttp\Psr7\Uri;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\Flow\Mvc\Routing\Dto\ResolveContext;
use Neos\Flow\Mvc\Routing\Dto\RouteParameters;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Flow\Utility\Now;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosRoutingBehaviourHooks\Behaviour\HostnameDimensionMatcher;
use Newland\NeosRoutingBehaviourHooks\Foundation\CustomDimensionResolution\DimensionResolutionContext;
use Newland\NeosRoutingBehaviourHooks\Foundation\RouteWrappingHelper;
use Newland\NeosRoutingBehaviourHooks\Utility\EnvironmentUtility;
use Newland\NeosRoutingBehaviourHooks\Utility\NamedDimensionCollection;

class HostnameDimensionMatcherTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var HostnameDimensionMatcher */
    private $subject;

    /** @var RouteWrappingHelper */
    private $helper;

    public function setUp(): void
    {
        parent::setUp();

        $this->subject = new HostnameDimensionMatcher();
        $this->inject($this->subject, 'hostMapping', new NamedDimensionCollection([
            'de.test.com' => [ 'language' => [ 'de' ] ],
            'en.test.com' => [ 'language' => [ 'en' ] ],
            'fr.test.com' => [ 'language' => [ 'fr' ] ],
        ]));

        $this->helper = new RouteWrappingHelper();
        $this->helper->setUri(new Uri('http://test.com/foo/bar'));
        $this->inject($this->subject, 'helper', $this->helper);
    }

    public function testAlwaysResolvesDimensionsToEmptyString()
    {
        $context = new DimensionResolutionContext();
        $this->assertEquals('', $this->subject->resolveDimensions($context, []));
        $this->assertEquals('', $this->subject->resolveDimensions($context, [ 'language' => [ 'de' ] ]));
        $this->assertEquals('', $this->subject->resolveDimensions($context, [ 'language' => [ 'en' ] ]));
        $this->assertEquals('', $this->subject->resolveDimensions($context, [ 'foo' => 'bar' ]));
    }

    public function testDoesNotResolveDimensionsWhenInBackend()
    {
        $this->helper->setUri(new Uri('https://de.test.com/neos/content'));
        $context = new DimensionResolutionContext();
        $this->assertEquals(null, $this->subject->resolveDimensions($context, []));
        $this->assertEquals(null, $this->subject->resolveDimensions($context, [ 'language' => [ 'de' ] ]));
        $this->assertEquals(null, $this->subject->resolveDimensions($context, [ 'language' => [ 'en' ] ]));
        $this->assertEquals(null, $this->subject->resolveDimensions($context, [ 'foo' => 'bar' ]));
    }

    public function testMatchesDimensionsFromHostname()
    {
        $uri = new Uri('https://de.test.com/foo/bar');
        $this->helper->setUri($uri);
        $this->assertEquals(
            [ 'language' => [ 'de' ] ],
            $this->subject->parseDimensions(new DimensionResolutionContext(), (string) $uri)
        );
    }

    public function testDoesNotMatchDimensionsWhenInBackend()
    {
        $uri = new Uri('https://de.test.com/neos/content');
        $this->helper->setUri($uri);
        $this->assertEquals(
            null,
            $this->subject->parseDimensions(new DimensionResolutionContext(), (string) $uri)
        );
    }

    public function testChangesHostInResolveContextDependingOnRequestedDimensions(): void
    {
        $context = new ResolveContext(
            new Uri('https://test.com'),
            [ 'node' => $this->initializeNode([ 'language' => [ 'en' ] ]) ],
            true,
            '',
            RouteParameters::createEmpty()
        );
        $newContext = $this->subject->modifyResolveContextBeforeResolving($context);
        $this->assertEquals('en.test.com', $newContext->getBaseUri()->getHost());
    }

    public function testDoesNotSelectHostOfIncorrectEnvironmentIfConfiguredInAnotherEnvironment(): void
    {
        $this->mockEnvironment('Production');
        $this->subject->setParameters([
            'hostsByEnvironment' => [
                'Production' => [
                    'en.foobar.com' => [ 'language' => 'en' ],
                    'de.foobar.com' => [ 'language' => 'de' ],
                ],
                'Preview' => [
                    'super-cool-english-domain.com' => [ 'language' => 'en' ],
                ],
            ],
        ]);

        // Requested domain is super similar to preview domain, however current environment
        // is production, so preview configuration should not be used.
        // See also: TBPOI-467
        $context = new ResolveContext(
            new Uri('https://super-cool-english-domain.de'),
            [ 'node' => $this->initializeNode([ 'language' => [ 'en' ] ]) ],
            true,
            '',
            RouteParameters::createEmpty()
        );

        $newContext = $this->subject->modifyResolveContextBeforeResolving($context);
        $this->assertEquals('en.foobar.com', $newContext->getBaseUri()->getHost());
    }

    public function testAllowsSettingDeprecatedHostConfiguration(): void
    {
        $this->mockEnvironment('Production');
        $this->subject->setParameters([
            'hosts' => [
                'en.foobar.com' => [ 'language' => 'en' ],
                'de.foobar.com' => [ 'language' => 'de' ],
            ],
        ]);

        $context = new ResolveContext(
            new Uri('https://foobar.com'),
            [ 'node' => $this->initializeNode([ 'language' => [ 'en' ] ]) ],
            true,
            '',
            RouteParameters::createEmpty()
        );

        $newContext = $this->subject->modifyResolveContextBeforeResolving($context);
        $this->assertEquals('en.foobar.com', $newContext->getBaseUri()->getHost());
    }

    private function initializeNode(array $targetDimensions, array $currentDimensions = null): Node
    {
        $currentDimensions = $currentDimensions ?? $targetDimensions;
        return new Node(
            new NodeData('/sites/foo/bar', new Workspace('live')),
            new ContentContext('live', new Now(), $currentDimensions, $targetDimensions, false, false, false)
        );
    }

    private function mockEnvironment(string $environmentToUse): void
    {
        $environment = $this->getMockBuilder(EnvironmentUtility::class)->getMock();
        $environment->method('getEnvironment')->willReturn($environmentToUse);
        $this->inject($this->subject, 'environmentUtility', $environment);
    }
}
