<?php declare(strict_types=1);

namespace Newland\NeosRoutingBehaviourHooks\Tests\Integration;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\ContentRepository\Domain\Service\NodeTypeManager;
use Neos\Flow\Http\Request;
use Neos\Flow\Http\Uri;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\Controller\Arguments;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Http\Response;
use Neos\Flow\Mvc\Routing\UriBuilder;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Flow\Utility\Now;
use Neos\Neos\Domain\Model\Domain;
use Neos\Neos\Domain\Model\Site;
use Neos\Neos\Domain\Repository\SiteRepository;
use Neos\Neos\Domain\Service\ContentContext;
use Neos\Neos\Service\LinkingService;
use Newland\NeosRoutingBehaviourHooks\Behaviour\HostnameDimensionMatcher;
use Newland\NeosRoutingBehaviourHooks\Foundation\RouteWrappingHelper;

class HostnameDimensionMatcherTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    public function setUp()
    {
        parent::setUp();

        $helper = new RouteWrappingHelper();
        $helper->setUri(new Uri('http://test.com/foo/bar'));
        $this->inject($helper, 'configuration', [ [
            'implementation' => HostnameDimensionMatcher::class,
            'parameters' => [
                'hostsByEnvironment' => [
                    'Testing' => [
                        'de.test.com' => [ 'language' => ['de'] ],
                        'en.test.com' => [ 'language' => ['en'] ],
                        'fr.test.com' => [ 'language' => ['fr'] ],
                    ]
                ]
            ]
        ] ]);
        $this->objectManager->setInstance(RouteWrappingHelper::class, $helper);
    }

    public function tearDown(): void
    {
        $this->objectManager->forgetInstance(RouteWrappingHelper::class);
        parent::tearDown();
    }

    /**
     * Neos LinkingService::createNodeUri does not use the `absolute` option of the Flow UriBuilder but
     * prepends domains manually leading to URLs generated here having duplicate host names in them sometimes.
     *
     * This manifests itself when the 2 following conditions are met:
     * 1. The host for the URL being built differs from the current host, forcing Flow to generate an absolute URL
     * 2. LinkingService::createNodeUri is supplied with the `$absolute` argument, leading to the additional prepending
     *    of another host.
     *
     * In this case, Flow is right and Neos is wrong and there is no denying that.
     * @SEE TBPOI-575
     */
    public function testDoesNotAddDuplicateHostIfResolvingNodeUriThroughNeosService(): void
    {
        $node = $this->initializeNode([ 'language' => 'fr' ], [ 'language' => [ 'fr' ] ]);
        $node->setProperty('uriPathSegment', 'les-foo-bar');
        $node->setNodeType(
            $this->objectManager
                ->get(NodeTypeManager::class)
                ->getNodeType('Neos.Neos:Document')
        );


        /** @var LinkingService $linkingService */
        $linkingService = $this->objectManager->get(LinkingService::class);

        $siteRepository = \Mockery::mock($this->objectManager->get(SiteRepository::class));
        $siteRepository->shouldReceive('findOneByNodeName')->andReturn($node->getContext()->getCurrentSite());

        $this->inject($linkingService, 'siteRepository', $siteRepository);
        $this->objectManager->setInstance(SiteRepository::class, $siteRepository);

        $context = new ControllerContext(
            new ActionRequest(Request::create(new Uri('http://test.com/foo/bar'))),
            new Response(),
            new Arguments(),
            new UriBuilder()
        );
        $uri = $linkingService->createNodeUri($context, $node, null, null, true);

        $this->assertRegExp('/^http:\/\/fr.test.com\/les-foo-bar.*$/', $uri);
    }

    private function initializeNode(array $targetDimensions, array $currentDimensions = null): Node
    {
        $site = new Site('foo');
        $domain = new Domain();
        $domain->setSite($site);
        $domain->setActive(true);
        $domain->setHostname('fr.test.com');
        $site->setPrimaryDomain($domain);

        $currentDimensions = $currentDimensions ?? $targetDimensions;
        $contentContext = new ContentContext('live', new Now(), $currentDimensions, $targetDimensions, false, false, false, $site);

        $siteNode = new Node(
            new NodeData('/sites/foo', new Workspace('live')),
            $contentContext
        );
        return new Node(
            new NodeData('/sites/foo/bar', new Workspace('live')),
            $contentContext
        );
    }

}
