<?php declare(strict_types=1);
namespace Newland\NeosRoutingBehaviourHooks\Utility;

use function Safe\json_encode;

class NamedDimensionCollection
{

    /**
     * @var array
     */
    private $dimensions = [];

    /**
     * @var array
     */
    private $keyHashMap = [];

    public function __construct(array $dimensions)
    {
        $this->dimensions = $this->normalizeDimensions($dimensions);
        $this->keyHashMap = $this->buildDimensionHashMap($this->dimensions);
    }

    public function dimension(string $key): ?array
    {
        return $this->dimensions[$key] ?? null;
    }

    /**
     * @param string[][] $values
     * @return string[]
     */
    public function keysForValues(array $values): array
    {
        $values = $this->normalizeDimension($values);
        return $this->keyHashMap[$this->dimensionHash($values)] ?? null;
    }

    public function keyForValuesMatchingReferenceMostClosely(array $values, string $reference): ?string
    {
        $keys = $this->keysForValues($values);
        usort($keys, function (string $a, string $b) use ($reference) {
            return levenshtein($a, $reference) - levenshtein($b, $reference);
        });
        return $keys[0] ?? null;
    }

    public function hasKeyForDimension(array $values): bool
    {
        $keys = $this->keysForValues($values);
        return \count($keys) > 0;
    }

    private function normalizeDimensions(array $dimensions): array
    {
        foreach ($dimensions as $key => $dimensionValues) {
            $dimensions[$key] = $this->normalizeDimension($dimensionValues);
        }
        return $dimensions;
    }

    private function normalizeDimension(array $dimensionValues)
    {
        foreach ($dimensionValues as $name => $value) {
            if (!is_array($value)) {
                $dimensionValues[$name] = [ $value ];
            }
        }
        return $dimensionValues;
    }

    private function buildDimensionHashMap(array $dimensions): array
    {
        $hashMap = [];
        foreach ($dimensions as $key => $values) {
            $hash = $this->dimensionHash($values);
            if (!array_key_exists($hash, $hashMap)) {
                $hashMap[$hash] = [];
            }
            $hashMap[$hash][] = $key;
        }
        return $hashMap;
    }

    private function dimensionHash(array $dimensionValues): string
    {
        return md5(json_encode($dimensionValues));
    }
}
