<?php declare(strict_types=1);
namespace Newland\NeosRoutingBehaviourHooks\Foundation;

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Neos\Flow\Mvc\Routing\Dto\ResolveContext;
use Neos\Flow\ObjectManagement\Exception\InvalidObjectException;
use Newland\NeosRoutingBehaviourHooks\Utility\StringUtility;
use Psr\Http\Message\UriInterface;

/**
 * @Flow\Scope("singleton")
 */
class RouteWrappingHelper
{

    /**
     * @var array
     * @Flow\InjectConfiguration("behaviours")
     */
    protected $configuration;

    /**
     * @var Behaviour[]
     */
    protected $behaviours;

    /**
     * @var UriInterface
     */
    protected $uri;

    /**
     * @return Behaviour[]
     */
    public function behaviours(): array
    {
        if ($this->behaviours === null) {
            $this->behaviours = array_map(
                function (array $config) {
                    return $this->initializeBehaviour($config);
                },
                $this->configuration ?? []
            );
        }

        return $this->behaviours;
    }

    public function isBackendRoute(): bool
    {
        if ($this->uri === null) {
            return false;
        }
        $path = $this->uri->getPath();
        if ($path{0} === '/') {
            $path = substr($path, 1);
        }

        return strpos($path, 'neos') === 0;
    }

    public function getUri(): UriInterface
    {
        return $this->uri;
    }

    public function setUri(UriInterface $uri): void
    {
        $this->uri = $uri;
    }

    private function initializeBehaviour(array $config): Behaviour
    {
        $class = $config['implementation'] ?? null;
        $parameters = $config['parameters'] ?? [];

        if ($class === null) {
            throw new InvalidConfigurationException('Behaviours must be defined with an `implementation` property');
        }

        if (!class_exists($class)) {
            throw new InvalidConfigurationException('Behaviour implementation ' . $class . ' was not found');
        }

        $instance = new $class();
        if (!($instance instanceof Behaviour)) {
            throw new InvalidObjectException(
                'Behaviour implementations must extend ' . Behaviour::class . ' but ' . $class . ' does not.'
            );
        }

        $instance->setParameters($parameters);
        return $instance;
    }
}
