<?php declare(strict_types=1);
namespace Newland\NeosRoutingBehaviourHooks\Behaviour;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Aop\JoinPointInterface;
use Neos\Flow\Mvc\Routing\Dto\ResolveContext;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosRoutingBehaviourHooks\Foundation\Behaviour;
use Newland\NeosRoutingBehaviourHooks\Foundation\CustomDimensionResolution\CustomDimensionResolution;
use Newland\NeosRoutingBehaviourHooks\Foundation\CustomDimensionResolution\DimensionResolutionContext;
use Newland\NeosRoutingBehaviourHooks\Foundation\IgnoreMatchedRoutes\IgnoreContext;
use Newland\NeosRoutingBehaviourHooks\Foundation\IgnoreMatchedRoutes\IgnoresMatchedNode;
use Newland\NeosRoutingBehaviourHooks\Foundation\ModifyResolveContext\ModifyResolveContext;
use Newland\NeosRoutingBehaviourHooks\Foundation\RouteWrappingHelper;
use Newland\NeosRoutingBehaviourHooks\Foundation\SeparateRouteCache\SeparateRouteCache;
use Newland\NeosRoutingBehaviourHooks\Utility\EnvironmentUtility;
use Newland\NeosRoutingBehaviourHooks\Utility\NamedDimensionCollection;
use Neos\Flow\Annotations as Flow;
use Newland\NeosRoutingBehaviourHooks\Utility\UrlUtility;

/**
 * @Flow\Aspect()
 */
class HostnameDimensionMatcher extends Behaviour implements
    IgnoresMatchedNode,
    ModifyResolveContext,
    CustomDimensionResolution,
    SeparateRouteCache
{

    /**
     * @var NamedDimensionCollection
     */
    protected $hostMapping;

    /**
     * @var RouteWrappingHelper
     * @Flow\Inject()
     */
    protected $helper;

    /**
     * @var EnvironmentUtility
     * @Flow\Inject()
     */
    protected $environmentUtility;


    public function setParameters(array $parameters): void
    {
        parent::setParameters($parameters);

        $environment = $this->environmentUtility->getEnvironment();
        $hostMappingConfig = $parameters['hostsByEnvironment'][$environment]
            ?? $this->legacyHostConfiguration($parameters)
            ?? [];
        $this->hostMapping = new NamedDimensionCollection($hostMappingConfig);
    }

    public function ignoreMatchedNode(IgnoreContext $context): bool
    {
        $disallow = $this->parameters['disallowOtherHostnames'] ?? false;
        if (!$disallow || $this->helper->isBackendRoute()) {
            return false;
        }

        $hostForNode = $this->hostForNode(
            $context->getNode(),
            $context->getRouteContext()->getHttpRequest()->getUri()->getHost()
        );
        if ($hostForNode === null) {
            return false;
        }

        return $this->helper->getCurrentHostname() !== $hostForNode;
    }

    public function modifyResolveContextBeforeResolving(ResolveContext $context): ResolveContext
    {
        $node = $context->getRouteValues()['node'] ?? null;

        if ($node === null || !($node instanceof Node)) {
            return $context;
        }

        $host = $this->hostForNode($node, $context->getBaseUri()->getHost());
        return $this->hostDiffers($context, $host) ? $this->applyHostnameToContext($context, (string) $host) : $context;
    }

    /**
     * Neos LinkingService has some special / custom hostname handling.
     * The combination of that and the forcing of absolute URLs in `applyHostnameToContext` leads to URLs that have
     * 2 hostname portions such as `https://foobar.com/https://foobar.com/some/path`.
     *
     * @Flow\Around("method(Neos\Neos\Service\LinkingService->createNodeUri())")
     */
    public function preventLinkingServiceFromAddingHostTwice(JoinPointInterface $joinPoint): string
    {
        return UrlUtility::stripOutDuplicateHostsInUrl($joinPoint->getAdviceChain()->proceed($joinPoint));
    }

    /**
     * Parses a request URI into an array of dimension values.
     * If an array is returned then this array will be used as dimension values,
     * If null is returned then this behaviour will have no impact.
     *
     * @param DimensionResolutionContext $context
     * @param string $requestPath
     * @return array|null
     */
    public function parseDimensions(DimensionResolutionContext $context, string $requestPath): ?array
    {
        $host = $this->helper->getCurrentHostname();
        if (!\is_string($host) || $this->helper->isBackendRoute()) {
            return null;
        }

        return $this->hostMapping->dimension($host);
    }

    /**
     * Resolves the given array of dimension values into a uri segment.
     * If a string is returned then this string will be used in the uri path,
     * If null is returned then this behaviour will have no impact.
     *
     * @param DimensionResolutionContext $context
     * @param array $dimensions
     * @return string|null
     */
    public function resolveDimensions(DimensionResolutionContext $context, array $dimensions): ?string
    {
        if ($this->helper->isBackendRoute()) {
            return null;
        }

        return '';
    }


    public function getRouteCacheTags(): array
    {
        return [
            'context' => $this->helper->isBackendRoute() ? 'backend' : 'frontend',
            'host' => $this->helper->getCurrentHostname(),
        ];
    }

    private function hostDiffers(ResolveContext $context, string $host = null)
    {
        return $host !== null && $context->getBaseUri()->getHost() !== $host;
    }

    private function hostForNode(Node $node, string $reference): ?string
    {
        /** @var ContentContext $context */
        $context = $node->getContext();

        if ($this->helper->isBackendRoute() || $context->isInBackend()) {
            $domain = $context->getCurrentSite()->getPrimaryDomain();
            if ($domain !== null) {
                return $domain->getHostname();
            }
        }

        $dimensions = $node->getDimensions()
            ?: $node->getContext()->getDimensions()
                ?: $node->getContext()->getTargetDimensionValues()
                    ?: [];

        return $this->hostMapping->keyForValuesMatchingReferenceMostClosely($dimensions, $reference);
    }

    private function applyHostnameToContext(ResolveContext $context, string $host): ResolveContext
    {
        $uri = $context->getBaseUri()->withHost($host);

        // Having `uriPathPrefix` in the ResolveContext will end up with URLs such as
        // `http://first-domain.com/prefix/http://second-domain.com`.
        $prefix = $context->getUriPathPrefix();
        if ($prefix && $prefix !== '/') {
            $uri = $uri->withPath($prefix . $uri->getPath());
        }

        return new ResolveContext($uri, $context->getRouteValues(), true, '/');
    }

    /** @deprecated Remove `hosts` config with 3.0.0 release */
    private function legacyHostConfiguration(array $parameters): ?array
    {
        return $parameters['hosts'] ?? null;
    }
}
