<?php
namespace Newland\NeosMetaTags\Extractor;

/*
 * This file is part of the "neos-meta-tags" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Neos\Flow\ObjectManagement\ObjectManager;
use Newland\NeosMetaTags\Domain\Model\CanonicalUrlGenerator;
use Newland\NeosMetaTags\Domain\Model\MetaData;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Newland\NeosMetaTags\Domain\Model\Scenario;
use Newland\Toubiz\Pulse\Domain\Model\ModelInterface;
use Newland\Toubiz\Pulse\Meta\CanonicalUrlGeneratorInterface;
use Newland\Toubiz\Pulse\Service\ItemResolver;
use Newland\Toubiz\Pulse\Service\ItemService;

class RequestExtractor extends BaseExtractor
{
    /**
     * @var ItemResolver
     * @Flow\Inject()
     */
    protected $itemResolver;
    /**
     * @var ItemService
     * @Flow\Inject()
     */
    protected $itemService;
    /**
     * @var ObjectManager
     * @Flow\Inject()
     */
    protected $objectManager;

    public function extractAll(Node $node, Scenario $scenario): MetaData
    {
        $record = $this->itemResolver->resolve(
            $scenario->getRecordIdentifier(),
            $scenario->getObjectType(),
            $node->getDimensions()['language'][0],
        );

        if ($record === null) {
            return new MetaData();
        }

        $title = $this->itemService->getPropertyWithFallback($record, $scenario->getPropertyMappingFor('title'));
        $description = strip_tags($this->extractMetaDescription($scenario, $record));
        $image = $this->itemService->getPropertyWithFallback($record, $scenario->getPropertyMappingFor('image'));

        $metaData = (new MetaData())
            ->setTitle($this->appendWebsite($title, $node))
            ->setDescription($description)
            ->setUrl($scenario->getUrl())
            ->setOgType($scenario->getOgType())
            ->setOgTitle($title)
            ->setOgDescription($description)
            ->setOgThumbnail($this->thumbnailService->getThumbnailDataFromExternalImage($image))
            ->setOgImageAltText($title)
            ->setTwitterType(MetaData::TWITTER_CARD_TYPE_DEFAULT)
            ->setTwitterTitle($title)
            ->setTwitterDescription($description)
            ->setTwitterImage($image)
            ->setTwitterCreator($this->twitterHandle);

        if ($scenario->getCanonicalUrlGenerator()) {
            $metaData->setCanonicalUrl(
                $this->generateCanonicalUrl($scenario, $record)
            );
        }

        return $metaData;
    }

    public function extractTitle(Node $node, Scenario $scenario, bool $appendWebsite = false): ?string
    {
        $record = $this->itemResolver->resolve(
            $scenario->getRecordIdentifier(),
            $scenario->getObjectType(),
            $node->getDimensions()['language'][0],
        );

        $recordTitle = $this->itemService->getPropertyWithFallback(
            $record,
            $scenario->getPropertyMappingFor('title')
        );

        if ($appendWebsite) {
            $recordTitle = $this->appendWebsite($recordTitle, $node);
        }

        return $recordTitle;
    }

    public function extractDescription(Node $node, Scenario $scenario): ?string
    {
        $record = $this->itemResolver->resolve(
            $scenario->getRecordIdentifier(),
            $scenario->getObjectType(),
            $node->getDimensions()['language'][0]
        );

        $description = $this->extractMetaDescription($scenario, $record);

        return (new MetaData())
            ->setDescription($description)
            ->getDescriptionTrimmed();
    }

    private function generateCanonicalUrl(Scenario $scenario, ModelInterface $record): ?string
    {
        $generatorClass = $scenario->getCanonicalUrlGenerator();
        if (!$generatorClass) {
            return null;
        }
        $generatorInstance = $this->objectManager->get($generatorClass);
        if (!($generatorInstance instanceof CanonicalUrlGeneratorInterface)) {
            throw new InvalidConfigurationException(
                sprintf(
                    'Configured canonical url generator must implement %s but %s does not.',
                    CanonicalUrlGenerator::class,
                    get_class($generatorInstance)
                )
            );
        }

        return $generatorInstance->generate($scenario, $record);
    }

    /** @param mixed $record */
    public function extractMetaDescription(Scenario $scenario, $record): ?string
    {
        return $this->itemService->getPropertyWithFallback($record, $scenario->getPropertyMappingFor('description'));
    }
}
