<?php declare(strict_types=1);

namespace Newland\NeosIcons\Controller;

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Http\Component\SetHeaderComponent;
use Neos\Flow\Mvc\Controller\ActionController;
use Neos\Flow\Utility\Now;
use Newland\NeosIcons\IconUtility;

class SvgController extends ActionController
{
    protected const MARKER_BASE_ICON = 'map-marker';
    protected $mimeType = 'image/svg+xml';
    protected $browserCacheLifetimeDays = 14;

    /**
     * @var IconUtility
     * @Flow\Inject()
     */
    protected $iconUtility;

    public function initializeAction()
    {
        $this->response->setContentType($this->mimeType);
        $this->response->setComponentParameter(
            SetHeaderComponent::class,
            'Cache-Control',
            'max-age=' . ($this->browserCacheLifetimeDays * 24 * 60 * 60)
        );
        $interval = sprintf('P%dD', $this->browserCacheLifetimeDays);
        $this->response->setComponentParameter(
            SetHeaderComponent::class,
            'Expires',
            (new Now())->add(new \DateInterval($interval))->format(DATE_RFC822)
        );
    }

    protected function callActionMethod(): void
    {
        try {
            parent::callActionMethod();
        } catch (\Throwable $throwable) {
            $message = sprintf(
                "%s\n%s:%s\n\n%s",
                $throwable->getMessage(),
                $throwable->getFile(),
                $throwable->getLine(),
                $throwable->getTraceAsString()
            );
            $content = $this->iconUtility->errorIcon($message)->toString();
            $this->response->setStatusCode(500);
            $this->response->setContent($content);
        }
    }

    public function mapMarkerAction(string $icon, string $foregroundColor, string $backgroundColor)
    {
        $symbol = $this->iconUtility->getIcon($icon)->fill($foregroundColor);
        return $this->iconUtility
            ->getIcon(static::MARKER_BASE_ICON)
            ->fill($backgroundColor)
            ->overlayIcon($symbol, 90, -75, 200, null)
            ->toString();
    }
}
