<?php declare(strict_types=1);

namespace Newland\NeosIcons;

use Neos\Cache\Frontend\FrontendInterface;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Eel\ProtectedContextAwareInterface;
use Neos\Flow\Cache\CacheManager;
use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\Routing\UriBuilder;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Neos\Flow\Annotations as Flow;
use function Safe\json_encode;

/**
 * @Flow\Scope("singleton")
 */
class IconUrlService implements ProtectedContextAwareInterface
{

    /**
     * @var UriBuilder
     * @Flow\Inject()
     */
    protected $uriBuilder;

    /**
     * @var ControllerContextFactory
     * @Flow\Inject()
     */
    protected $controllerContextFactory;

    /**
     * @var ConfigurationManager
     * @Flow\Inject()
     */
    protected $configurationManager;

    /**
     * @var FrontendInterface
     */
    protected $cache;
    public function injectCache(CacheManager $cacheManager): void
    {
        $this->cache = $cacheManager->getCache('Newland_NeosIcons_IconUrls');
    }

    public function buildMarkerIconUrl(
        array $style,
        ActionRequest $request,
        string $defaultConfigurationPath = 'Newland.NeosIcons.defaults.markerIcon'
    ): string {
        $key = md5(json_encode($style) . $request->getHttpRequest()->getUri()->getHost() . $defaultConfigurationPath);
        if ($this->cache->has($key)) {
            return $this->cache->get($key);
        }

        $defaultIconSettings = $this->configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            $defaultConfigurationPath
        );

        $icon = $style['icon']
            ?? $defaultIconSettings['icon']
            ?? 'circle-solid';

        $backgroundColor = $style['backgroundColor']
            ?? $defaultIconSettings['backgroundColor']
            ?? '#888';

        $foregroundColor = $style['color']
            ?? $defaultIconSettings['color']
            ?? '#fff';

        $this->uriBuilder->setRequest($request);
        $uri = $this->uriBuilder->build(
            [
                '@package' => 'Newland.NeosIcons',
                '@controller' => 'svg',
                '@action' => 'mapMarker',
                '@format' => 'svg',
                'icon' => $icon,
                'backgroundColor' => $backgroundColor,
                'foregroundColor' => $foregroundColor,
            ]
        );

        $this->cache->set($key, $uri);
        return $uri;
    }

    public function buildMarkerIconUrlFromNode(
        NodeInterface $articleNode,
        array $style,
        string $defaultConfigurationPath = 'Newland.NeosIcons.defaults.markerIcon'
    ): string {
        return $this->buildMarkerIconUrl(
            $style,
            $this->controllerContextFactory->initializeFakeActionRequest($articleNode),
            $defaultConfigurationPath
        );
    }

    public function allowsCallOfMethod($methodName)
    {
        return $methodName === 'buildMarkerIconUrlFromNode';
    }
}
