<?php declare(strict_types=1);

namespace Newland\NeosFiltering\Items;

use Doctrine\ORM\Query\Expr;
use Newland\NeosFiltering\Contract\DataSourcedFilterItem;
use Newland\NeosFiltering\Contract\Expression;
use Newland\NeosFiltering\Contract\FilterItem;
use Newland\NeosFiltering\Contract\FilterItemCommon;
use Newland\NeosFiltering\Contract\HasCombinationSettings;
use Newland\NeosFiltering\Contract\HasDataSource;
use Newland\NeosFiltering\Contract\HasQueryString;
use Newland\NeosFiltering\Contract\HasVisibleState;
use Newland\NeosFiltering\Contract\HasTitle;
use Newland\NeosFiltering\Contract\NeedsDatabaseColumn;
use Newland\NeosFiltering\Contract\QueryBoundFilterItem;
use Newland\NeosFiltering\Contract\StatusIndicatingFilterItem;
use Newland\NeosFiltering\Contract\StatusIndicator;
use Newland\NeosFiltering\Contract\TitledFilterItem;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;

/**
 * A collection of checkboxes that allow the user to select multiple values for the
 * same identifier.
 *
 * The state must be set to an array containing strings for all selected options.
 *
 * Additional to the usual data source attributes `label` and `value` a data source may also
 * specify `queryString`. If an option has a `queryString` that string will be used in the
 * query.
 *
 * @example
 * $field = new CheckboxCollection();
 * $field->setQueryString('tags');
 * $field->setTitle('Article Tags');
 * $field->setDatabaseColumn('entity.tag');
 * $field->setDataSource(new DirectDataSource([
 *      [ 'label' => 'Foo', value => 'foo' ],
 *      [ 'label' => 'Bar', 'value' => 'bar', 'queryString' => 'uebernachten' ],
 *      [ 'label' => 'Baz', 'value' => 'baz' ],
 * ]));
 * $field->setState([ 'foo', 'bar' ]);
 */
class CheckboxList implements
    FilterItem,
    QueryBoundFilterItem,
    DataSourcedFilterItem,
    TitledFilterItem,
    StatusIndicatingFilterItem
{
    use FilterItemCommon,
        HasTitle,
        HasVisibleState,
        NeedsDatabaseColumn,
        HasDataSource,
        HasCombinationSettings,
        HasQueryString {
        HasDataSource::setState insteadof FilterItemCommon;
    }

    public const MAX_ITEMS_BEFORE_EXPANDABLE = 3;

    public function queryExpression(Expr $expr): Expression
    {
        $this->throwIfNoDatabaseColumnDeclared();
        $this->throwIfNoQueryString();
        $state = $this->getStateValuesForQuery();

        if (\count($state) === 0) {
            return Expression::empty();
        }

        if (\count($state) === 1) {
            return Expression::where($expr->eq($this->databaseColumn, $expr->literal(array_values($state)[0])));
        }

        $expression = Expression::where($expr->in($this->databaseColumn, $state));
        if ($this->combine === static::$AND) {
            $expression->having = $expr->gte($expr->countDistinct($this->databaseColumn), \count($state));
        }
        return $expression;
    }

    protected function getStateValuesForQuery(): array
    {
        return $this->getStateValues();
    }

    public function render(RenderingContextInterface $renderingContext)
    {
        $this->configuration['maxItemsBeforeExpandable'] = $this->configuration['maxItemsBeforeExpandable']
            ?? static::MAX_ITEMS_BEFORE_EXPANDABLE;
        $view = $this->initializeView();
        $view->assignMultiple(
            [
                'options' => $this->getDataFromDataSource(),
                'stateValues' => $this->getStateValues(),
            ]
        );
        return $view->render('CheckboxList');
    }

    /** @return StatusIndicator[] */
    public function getStatusIndicators(): array
    {
        return array_map(
            function (array $item) {
                return StatusIndicator::fromOption($item);
            },
            (array) $this->state
        );
    }
}
