<?php declare(strict_types=1);

namespace Newland\NeosFiltering\Items;

use Doctrine\ORM\Query\Expr;
use Newland\NeosFiltering\Contract\Expression;
use Newland\NeosFiltering\Contract\FilterItem;
use TYPO3Fluid\Fluid\Core\Parser\SyntaxTree\NodeInterface;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;

/**
 * Filter item that obstructs some access to the contained item.
 * This filter item will pass through all calls to the obstructed items but
 * - It will not render anything
 * - Because it does not implement the `QueryBoundFilterItem` interface all query bindings of
 *   the obstructed item will be ignored.
 *
 * This item can be used to preserver the functionality of a filter item while removing it
 * from the visual output.
 */
class ObstructionWrapper implements FilterItem
{

    /** @var FilterItem */
    protected $obstructed;

    public function __construct(FilterItem $obstructed)
    {
        $this->obstructed = $obstructed;
    }

    public function __call(string $method, array $arguments)
    {
        return $this->obstructed->{$method}(...$arguments);
    }

    public function __clone()
    {
        $this->obstructed = clone $this->obstructed;
    }

    public function queryExpression(Expr $expr): Expression
    {
        return $this->obstructed->queryExpression($expr);
    }

    public function setState($value): void
    {
        $this->obstructed->setState($value);
    }

    public function getName()
    {
        return $this->obstructed->getName();
    }

    public function setName($name)
    {
        $this->obstructed->setName($name);
    }

    public function setNode(NodeInterface $node)
    {
        $this->obstructed->setNode($node);
    }

    public function getNode()
    {
        return $this->obstructed->getNode();
    }

    public function render(RenderingContextInterface $renderingContext)
    {
        return null;
    }

    public function getId(): string
    {
        return $this->obstructed->getId();
    }

    public function setRoot(Root $root): void
    {
        $this->obstructed->setRoot($root);
    }

    public function getRoot(): Root
    {
        return $this->obstructed->getRoot();
    }

    public function setConfiguration(array $configuration): void
    {
        $this->obstructed->setConfiguration($configuration);
    }

    public function getObstructed(): FilterItem
    {
        return $this->obstructed;
    }

    public function getConfiguration(): array
    {
        return $this->obstructed->getConfiguration();
    }

    public function getState()
    {
        return $this->obstructed->getState();
    }
}
