<?php declare(strict_types=1);

namespace Newland\NeosFiltering\Tests\Unit\Items;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Newland\NeosFiltering\DataSource\DirectDataSource;
use Newland\NeosFiltering\Items\SelectBox;
use Newland\NeosFiltering\Tests\Factory\ExampleEntityFactory;
use Newland\NeosFiltering\Tests\Fixture\Domain\Model\ExampleEntity;

class SelectBoxTest extends ItemTestCase
{
    /** @var SelectBox */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();

        $this->subject = new SelectBox();
        $this->subject->setDatabaseColumn('entity.database_column');
        $this->subject->setQueryString('query_string');
        $this->subject->setTitle('Title');
        $this->subject->setDataSource(new DirectDataSource());
        $this->subject->setDataSourceArguments(
            [
                'options' => [
                    [ 'label' => 'Foo', 'value' => 'foo' ],
                    [ 'label' => 'Bar', 'value' => 'bar' ],
                    [ 'label' => 'Baz', 'value' => 'baz' ],
                ],
            ]
        );
    }

    public function testRendersSelect(): void
    {
        $result = $this->renderSubject();
        $this->assertContains('<select', $result);
        $this->assertContains('name="query_string"', $result);
    }

    public function testRendersOptionsForEachDataItem(): void
    {
        $result = $this->renderSubject();
        $this->assertContains('<option value="foo"', $result);
        $this->assertContains('<option value="bar"', $result);
        $this->assertContains('<option value="baz"', $result);
    }

    public function testSelectsOptionAccordingToState(): void
    {
        $this->subject->setState('foo');
        $result = $this->renderSubject();
        $this->assertContains('<option value="foo" selected', $result);
    }

    public function testUsesLabelsFromDataSource(): void
    {
        $this->subject->setDataSource(new DirectDataSource());
        $this->subject->setDataSourceArguments(
            [
                'options' => [
                    [ 'label' => 'LabelFirst', 'value' => 'first' ],
                    [ 'label' => 'LabelSecond', 'value' => 'second' ],
                    [ 'label' => 'LabelThird', 'value' => 'third' ],
                ],
            ]
        );


        $rendered = $this->renderSubject();
        $this->assertContains('LabelFirst', $rendered);
        $this->assertContains('LabelSecond', $rendered);
        $this->assertContains('LabelThird', $rendered);
    }

    public function testRendersTitle(): void
    {
        $this->subject->setTitle('THIS IS A TITLE');
        $this->assertContains('THIS IS A TITLE', $this->renderSubject());
    }

    public function testDoesNotCreateExpressionIfStateEmpty(): void
    {
        $this->subject->setState('');
        $expression = $this->subject->queryExpression($this->entityManager->getExpressionBuilder());
        $this->assertNull($expression->having);
        $this->assertNull($expression->where);
    }

    public function testFindsSpecificThingIfStateGiven(): void
    {
        (new ExampleEntityFactory($this->objectManager))->createMultiple(2, [ 'tag' => 'foo' ]);
        (new ExampleEntityFactory($this->objectManager))->createMultiple(2, [ 'tag' => 'bar' ]);

        $this->subject->setDatabaseColumn('entity.tag');
        $this->subject->setState('foo');
        $this->assertCount(2, $this->queryForSubject()->getQuery()->execute());
    }
}
