<?php declare(strict_types=1);

namespace Newland\NeosFiltering\Tests\Unit\DataSource;

use Neos\Flow\I18n\Translator;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\NeosFiltering\DataSource\DirectDataSource;
use PhpParser\Node\Scalar\MagicConst\Dir;

class DirectDataSourceTest extends FunctionalTestCase
{

    /** @var DirectDataSource */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = new DirectDataSource();
    }

    public function testReturnsOptionsPassedAsArgument(): void
    {
        $result = $this->subject->getData(
            null,
            [
                'options' => [
                    [ 'label' => 'Foo', 'value' => 'foo' ],
                    [ 'label' => 'Bar', 'value' => 'bar' ],
                ],
            ]
        );

        $this->assertCount(2, $result);
        $this->assertEquals('foo', $result[0]['value']);
        $this->assertEquals('bar', $result[1]['value']);
    }

    public function testFetchesLanguageLabelForOptions(): void
    {
        $translator = $this->createMock(Translator::class);
        $translator->expects($this->once())
            ->method('translateById')
            ->with('foo', [], null, null, 'View/Filter', 'Newland.MyTheme')
            ->willReturn('This is a test');

        $this->inject($this->subject, 'translator', $translator);

        $result = $this->subject->getData(
            null,
            [
                'i18n' => [
                    'package' => 'Newland.MyTheme',
                    'source' => 'View/Filter',
                ],
                'options' => [
                    [ 'value' => 'foo' ],
                ],
            ]
        );

        $this->assertEquals('This is a test', $result[0]['label']);
    }

    public function testAllowsSpecifyingCustomLanguageLabelIdForOption(): void
    {
        $translator = $this->createMock(Translator::class);
        $translator->expects($this->once())
            ->method('translateById')
            ->with('custom.property.foo', [], null, null, 'View/Filter', 'Newland.MyTheme')
            ->willReturn('Custom property works');

        $this->inject($this->subject, 'translator', $translator);

        $result = $this->subject->getData(
            null,
            [
                'i18n' => [
                    'package' => 'Newland.MyTheme',
                    'source' => 'View/Filter',
                ],
                'options' => [
                    [ 'value' => 'foo', 'i18n' => [ 'id' => 'custom.property.foo' ] ],
                ],
            ]
        );

        $this->assertEquals('Custom property works', $result[0]['label']);
    }

    public function testAllowsSpecifyingCustomI18nConfigurationForOption(): void
    {
        $translator = $this->createMock(Translator::class);
        $translator->expects($this->once())
            ->method('translateById')
            ->with('custom.property.foo', [], null, null, 'View/Filter/Custom', 'Newland.MyOtherTheme')
            ->willReturn('Custom property works');

        $this->inject($this->subject, 'translator', $translator);

        $result = $this->subject->getData(
            null,
            [
                'i18n' => [
                    'package' => 'Newland.MyTheme',
                    'source' => 'View/Filter',
                ],
                'options' => [
                    [
                        'value' => 'foo',
                        'i18n' => [
                            'id' => 'custom.property.foo',
                            'package' => 'Newland.MyOtherTheme',
                            'source' => 'View/Filter/Custom',
                        ],
                    ],
                ],
            ]
        );

        $this->assertEquals('Custom property works', $result[0]['label']);
    }

}
