<?php declare(strict_types=1);

namespace Newland\NeosFiltering\Items;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\Exception\InfiniteLoopException;
use Neos\Flow\Mvc\Routing\UriBuilder;
use Newland\NeosFiltering\Contract\CompositeFilterItem;
use Newland\NeosFiltering\Contract\FilterItem;
use Newland\NeosFiltering\Contract\QueryBoundFilterItem;
use Newland\NeosFiltering\Contract\StatusIndicatingFilterItem;
use Newland\NeosFiltering\Contract\StatusIndicator;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;
use Neos\Flow\Annotations as Flow;

/**
 * The root object of a whole filter.
 * Additionally to rendering the frame around a filter this class has a bit more
 * functionality:
 *
 * - Applies the expressions returned by it's children to a given query builder.
 * - Maintains a mapping of query string => filter item which is used to set the state on
 *   it's children.
 */
class Root extends Group implements StatusIndicatingFilterItem
{

    /** @var FilterItem[] */
    protected $queryStringMapping = [];

    /** @var array */
    protected $joins = [];

    /** @var string */
    protected $actionUri;

    /**
     * @var UriBuilder
     * @Flow\Inject()
     */
    protected $uriBuilder;

    /**
     * @var EntityManagerInterface
     * @Flow\Inject()
     */
    protected $entityManager;

    /** @var array */
    protected $formAttributes = [];

    public function applyToQuery(QueryBuilder $query): QueryBuilder
    {
        $alias = $query->getRootAliases()[0];
        $entityClass = $query->getDQLPart('from')[0]->getFrom();
        $metadata = $this->entityManager->getMetadataFactory()->getMetadataFor($entityClass);
        $identifierFieldName = $metadata->getIdentifierFieldNames()[0];

        $query->groupBy(sprintf('%s.%s', $alias, $identifierFieldName));

        foreach ($this->joins as $join) {
            if ($this->hasJoin($query, $join['alias'])) {
                continue;
            }

            switch ($join['type'] ?? null) {
                case 'inner':
                    $query->innerJoin($join['column'], $join['alias']);
                    break;
                case 'left':
                default:
                    $query->leftJoin($join['column'], $join['alias']);
                    break;
            }
        }

        $expression = $this->queryExpression($query->expr());
        if ($expression->where) {
            $query->where($expression->where);
        }
        if ($expression->having) {
            $query->having($expression->having);
        }
        return $query;
    }

    public function setJoins(array $joins): void
    {
        $this->joins = $joins;
    }

    public function setState($state): void
    {
        if (!\is_array($state)) {
            throw new \InvalidArgumentException('Given state must be an array.');
        }

        foreach ($state as $key => $value) {
            if (array_key_exists($key, $this->queryStringMapping)) {
                $this->queryStringMapping[$key]->setState($value);
            }
        }
    }

    public function addItem(FilterItem $item): void
    {
        foreach ($this->flattenChildren($item) as $child) {
            if ($child instanceof QueryBoundFilterItem) {
                $this->queryStringMapping[$child->getQueryString()] = $child;
            }
        }
        parent::addItem($item);
    }

    public function render(RenderingContextInterface $renderingContext)
    {
        if ($this->actionUri === null) {
            throw new \InvalidArgumentException('Please set an action uri on the filter root.');
        }

        $attributes = '';
        foreach ($this->formAttributes as $key => $value) {
            $attributes .= sprintf(' %s="%s" ', $key, $value);
        }

        $view = $this->initializeView();
        $view->assignMultiple(
            [
                'items' => $this->items,
                'actionUri' => $this->actionUri,
                'statusIndicators' => $this->getStatusIndicators(),
                'formAttributes' => $attributes
            ]
        );
        return $view->render('Root');
    }

    public function fillStateFromRootRequestArguments(ActionRequest $request): self
    {
        $this->setState($request->getMainRequest()->getArguments());
        return $this;
    }

    private function flattenChildren(FilterItem $item): \Generator
    {
        yield $item;
        if (!($item instanceof CompositeFilterItem)) {
            return;
        }

        /** @var CompositeFilterItem[] $queue */
        $queue = [ $item ];
        $i = 0;
        // phpcs:ignore
        while ($item = array_pop($queue)) {
            foreach ($item->getChildrenItems() as $child) {
                yield $child;
                if ($child instanceof CompositeFilterItem) {
                    $queue[] = $child;
                }
                if (++$i > 1000) {
                    throw new InfiniteLoopException(
                        'Stopped flattening children after 1000 iterations. Suspected loop in tree'
                    );
                }
            }
        }
    }

    public function setActionUri(string $actionUri): self
    {
        $this->actionUri = $actionUri;
        return $this;
    }


    public function setAction(array $arguments, ActionRequest $request, NodeInterface $node = null): self
    {
        $this->uriBuilder->setRequest($request);
        $this->actionUri = $this->uriBuilder
            ->reset()
            ->setCreateAbsoluteUri(true)
            ->setArguments([ $request->getArgumentNamespace() => $arguments ])
            ->uriFor(
                'show',
                [ 'node' => $node ],
                'Frontend\Node',
                'Neos.Neos'
            );

        return $this;
    }

    private function hasJoin(QueryBuilder $query, string $alias): bool
    {
        foreach ($query->getDQLPart('join')['entity'] ?? [] as $join) {
            /** @var Join $join */
            if ($join->getAlias() === $alias) {
                return true;
            }
        }
        return false;
    }

    /** @return StatusIndicator[] */
    public function getStatusIndicators(): array
    {
        $indicators = [ [] ];
        foreach ($this->flattenChildren($this) as $child) {
            if ($child instanceof StatusIndicatingFilterItem && $child !== $this) {
                $indicators[] = $child->getStatusIndicators();
            }
        }
        return array_merge(...$indicators);
    }

    public function setFormAttributes(array $formAttributes): void
    {
        $this->formAttributes = $formAttributes;
    }
}
