<?php declare(strict_types=1);

namespace Newland\NeosFiltering\Items;

use Doctrine\ORM\Query\Expr;
use Newland\NeosFiltering\Contract\DataSourcedFilterItem;
use Newland\NeosFiltering\Contract\Expression;
use Newland\NeosFiltering\Contract\FilterItem;
use Newland\NeosFiltering\Contract\FilterItemCommon;
use Newland\NeosFiltering\Contract\HasCombinationSettings;
use Newland\NeosFiltering\Contract\HasDataSource;
use Newland\NeosFiltering\Contract\HasQueryString;
use Newland\NeosFiltering\Contract\HasTitle;
use Newland\NeosFiltering\Contract\NeedsDatabaseColumn;
use Newland\NeosFiltering\Contract\QueryBoundFilterItem;
use Newland\NeosFiltering\Contract\StatusIndicatingFilterItem;
use Newland\NeosFiltering\Contract\StatusIndicator;
use Newland\NeosFiltering\Contract\TitledFilterItem;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;

/**
 * A collection of checkboxes that allow the user to select multiple values for the
 * same identifier.
 *
 * The state must be set to an array containing strings for all selected options.
 *
 * @example
 * $field = new CheckboxCollection();
 * $field->setQueryString('tags');
 * $field->setTitle('Article Tags');
 * $field->setDatabaseColumn('entity.tag');
 * $field->setDataSource(new DirectDataSource([
 *      [ 'label' => 'Foo', value => 'foo' ],
 *      [ 'label' => 'Bar', 'value' => 'bar' ],
 *      [ 'label' => 'Baz', 'value' => 'baz' ],
 * ]));
 * $field->setState([ 'foo', 'bar' ]);
 */
class CheckboxList implements
    FilterItem,
    QueryBoundFilterItem,
    DataSourcedFilterItem,
    TitledFilterItem,
    StatusIndicatingFilterItem
{
    use FilterItemCommon,
        HasTitle,
        NeedsDatabaseColumn,
        HasDataSource,
        HasCombinationSettings,
        HasQueryString;

    public function setState($state): void
    {
        $validValues = array_map(
            function ($item) {
                return $item['value'];
            },
            $this->getDataFromDataSource()
        );

        $this->state = array_filter(
            (array) $state,
            function ($item) use ($validValues) {
                return \in_array($item, $validValues);
            }
        );
    }

    public function queryExpression(Expr $expr): Expression
    {
        $this->throwIfNoDatabaseColumnDeclared();
        $this->throwIfNoQueryString();
        $state = (array) $this->state;

        if (\count($state) === 0) {
            return Expression::empty();
        }

        if (\count($state) === 1) {
            return Expression::where($expr->eq($this->databaseColumn, $expr->literal($state[0])));
        }

        $expression = Expression::where($expr->in($this->databaseColumn, $state));
        if ($this->combine === static::$AND) {
            $expression->having = $expr->gte($expr->countDistinct($this->databaseColumn), \count($state));
        }
        return $expression;
    }

    public function render(RenderingContextInterface $renderingContext)
    {
        $view = $this->initializeView();
        $view->assign('options', $this->getDataFromDataSource());
        return $view->render('CheckboxList');
    }

    /** @return StatusIndicator[] */
    public function getStatusIndicators(): array
    {
        return array_map(
            function (array $item) {
                return StatusIndicator::fromOption($item);
            },
            $this->getSelectedOptions()
        );
    }

    private function getSelectedOptions(): array
    {
        $selected = [];
        $options = $this->getDataFromDataSource();

        foreach ($this->state ?? [] as $value) {
            foreach ($options as $option) {
                if ($value === $option['value']) {
                    $selected[] = $option;
                }
            }
        }

        return $selected;
    }
}
