<?php declare(strict_types=1);

namespace Newland\NeosFiltering\RangeSource;

use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\EntityManagerInterface;
use Neos\Flow\Annotations as Flow;

/**
 * Range Source that get's the minimum and maximum values from a database field.
 */
class SimpleDatabaseRangeSource implements RangeSource
{

    /** @var string */
    protected $class;

    /** @var string */
    protected $column;

    /** @var string */
    protected $alias;

    /** @var string|null */
    protected $where;

    /** @var array */
    protected $minAndMax;

    /**
     * @var EntityManagerInterface
     * @Flow\Inject()
     */
    protected $entityManager;

    public function setArguments(array $arguments): void
    {
        $this->class = $arguments['class'];
        $this->column = $arguments['column'];
        $this->alias = $arguments['alias'] ?? 'entity';
        $this->where = $arguments['where'] ?? null;
    }

    public function min(): ?float
    {
        $this->initialize();
        $min = $this->minAndMax['min'] ?? null;
        return $min !== null ? (float) $min : null;
    }

    public function max(): ?float
    {
        $this->initialize();
        $max = $this->minAndMax['max'] ?? null;
        return $max !== null ? (float) $max : null;
    }

    private function initialize(): void
    {
        if ($this->minAndMax !== null) {
            return;
        }

        $column = sprintf('%s.%s', $this->alias, $this->column);
        $query = $this->entityManager->createQueryBuilder()
            ->select(
                sprintf('MIN(%s) AS min', $column),
                sprintf('MAX(%s) AS max', $column)
            )->from($this->class, $this->alias);

        if ($this->where !== null) {
            $query->where($this->where);
        }

        $result = $query->getQuery()->execute([], AbstractQuery::HYDRATE_ARRAY);
        $this->minAndMax = $result[0] ?? [ 'min' => null, 'max' => null ];
    }
}
