<?php declare(strict_types=1);
namespace Newland\NeosFiltering\Contract;

/*
 * This file is part of the "neos-filtering" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Neos\Service\DataSource\DataSourceInterface;
use function Safe\preg_replace;

/**
 * Default implementation for the `DataSourcedFilterItem` interface.
 *
 * Note: This interface only provides a common implementation - you must explicitly implement
 *       the `DataSourcedFilterItem` interface if you want to mark your filter item as data sourced.
 *
 * Additional to the regular data source value (`label` & `value`) a item from a data source may
 * also contain `queryString` to specify a human readable string.
 *
 * Note: When using this trait together with `FilterItemCommon` you will have to explicitly specify
 *      that you want to use `HasDataSource::setState` instead of the one in `FilterItemCommon`:
 * ```
 * use FilterItemCommon, HasDataSource {
 *      use FilterItemCommon::setState instead of FilterItemCommon;
 * }
 * ```
 *
 * @see DataSourcedFilterItem
 */
trait HasDataSource
{

    /** @var DataSourceInterface */
    protected $dataSource;

    /** @var array */
    protected $dataSourceArguments = [];

    public function setDataSource(DataSourceInterface $dataSource): void
    {
        $this->dataSource = $dataSource;
    }

    public function getDataSource(): ?DataSourceInterface
    {
        return $this->dataSource;
    }

    public function setDataSourceArguments(array $arguments): void
    {
        $this->dataSourceArguments = $arguments;
    }

    public function getDataSourceArguments(): array
    {
        return $this->dataSourceArguments;
    }

    public function setState($state): void
    {
        $this->state = $this->extractValidDataSourceOptionsBasedOnStateString((array) $state);
    }

    protected function extractValidDataSourceOptionsBasedOnStateString(array $state): array
    {
        $state = array_filter(
            $this->getDataFromDataSource(),
            function (array $row) use ($state) {
                $queryString = $row['queryString'] ?? null;
                if ($queryString && \in_array($queryString, $state, true)) {
                    return true;
                }
                return \in_array($row['value'], $state, true);
            }
        );

        return array_values($state);
    }

    protected function getDataSourceRowsForCurrentState(): array
    {
        $state = (array) $this->getState();
        return array_filter(
            $this->getDataFromDataSource(),
            function (array $row) use ($state) {
                $valueToMatch = $row['queryString'] ?? $row['value'];
                return \in_array($valueToMatch, $state, true);
            }
        );
    }

    public function getDataFromDataSource(): array
    {
        $node = null;
        if ($this instanceof FilterItem) {
            $node = $this->getRoot()->getPluginNode();
        }

        $data = $this->dataSource->getData($node, $this->dataSourceArguments ?? []);
        return array_map(
            function (array $item) {
                $niceValue = preg_replace('/\W/', '-', $item['value']);
                $niceValue = \is_array($niceValue) ? implode('-', $niceValue) : $niceValue;
                $item['id'] = sprintf('%s-%s', $this->getId(), $niceValue);
                return $item;
            },
            $data
        );
    }

    protected function getStateValues(): array
    {
        return array_column((array) $this->state, 'value');
    }

    public function getHiddenFieldsIfObstructed(): array
    {
        if (!\is_array($this->state)) {
            return [];
        }

        $fields = [];
        foreach ($this->state as $index => $state) {
            $name = sprintf('%s[%d]', $this->queryString, $index);
            $fields[$name] = $state['value'];
        }
        return $fields;
    }
}
