<?php declare(strict_types=1);

namespace Newland\NeosFiltering\Tests\Unit\Items;

use Newland\NeosFiltering\Items\Range;
use Newland\NeosFiltering\RangeSource\DirectRangeSource;
use Newland\NeosFiltering\Tests\Factory\ExampleEntityFactory;

class RangeTest extends ItemTestCase
{
    /** @var Range */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = new Range();
        $this->subject->setTitle('My Range');
        $this->subject->setDatabaseColumn('entity.number');
        $this->subject->setQueryString('range_query_string');

        $source = new DirectRangeSource();
        $source->setArguments([ 'min' => 0, 'max' => 100 ]);
        $this->subject->setRangeSource($source);
    }

    public function testDisplaysTitle(): void
    {
        $this->assertContains('My Range', $this->renderSubject());
    }

    public function testRendersTwoInputFieldsForMinAndMax(): void
    {
        $rendered = $this->renderSubject();
        $this->assertContainsCount('<input', 2, $rendered);
        $this->assertContains('name="range_query_string[min]', $rendered);
        $this->assertContains('name="range_query_string[max]', $rendered);
    }

    public function testAppliesCurrentStateAsDefaultValuesToCurrentState(): void
    {
        $this->subject->setState([ 'min' => 20, 'max' => 80 ]);
        $rendered = $this->renderSubject();
        $this->assertContains('value="20"', $rendered);
        $this->assertContains('value="80"', $rendered);


        $this->subject->setState([ 'min' => '40', 'max' => '60' ]);
        $rendered = $this->renderSubject();
        $this->assertContains('value="40"', $rendered);
        $this->assertContains('value="60"', $rendered);
    }

    public function testUsesMinAndMaxValuesProvidedByRangeSourceForFields(): void
    {
        $rendered = $this->renderSubject();
        $this->assertContainsCount('min="0"', 2, $rendered);
        $this->assertContainsCount('max="100"', 2, $rendered);
    }

    public function testUsesMinAndMaxValuesProvidedByRangeSourceAsPlaceholder(): void
    {
        $rendered = $this->renderSubject();
        $this->assertContains('placeholder="0"', $rendered);
        $this->assertContains('placeholder="100"', $rendered);
    }

    public function testDoesNotAddDefaultValuesIfNoStateAvailable(): void
    {
        $this->subject->setState(null);
        $this->assertNotContains('value=', $this->renderSubject());
        $this->subject->setState([]);
        $this->assertNotContains('value=', $this->renderSubject());
    }

    /** @dataProvider provideInvalidState */
    public function testDoesNotAddDefaultValuesIfStateInvalid($state): void
    {
        $this->subject->setState($state);
        $this->assertNotContains('value=', $this->renderSubject());
    }

    /** @dataProvider providesWeirdRanges */
    public function testNormalizesValuesIfStateDoesNotFormARangeThatMakesSense($state, $min, $max): void
    {
        $this->subject->setState($state);
        $rendered = $this->renderSubject();
        $this->assertContains(sprintf('value="%d"', $min), $rendered);
        $this->assertContains(sprintf('value="%d"', $max), $rendered);
    }

    public function testFiltersForRangeInclusive(): void
    {
        $factory = new ExampleEntityFactory($this->objectManager);
        $five = $factory->create([ 'number' => 5 ]);
        $ten = $factory->create([ 'number' => 10 ]);
        $fifteen = $factory->create([ 'number' => 15 ]);
        $twenty = $factory->create([ 'number' => 20 ]);
        $twentyFive = $factory->create([ 'number' => 25 ]);

        $this->subject->setState([ 'min' => 10, 'max' => '20' ]);
        $ids = array_map(
            function($item) { return $item->uuid; },
            $this->queryForSubject()->getQuery()->execute()
        );

        $this->assertNotContains($five->uuid, $ids, 'Range between 10 and 20 should not find 5');
        $this->assertContains($ten->uuid, $ids, 'Range between 10 and 20 should find 10');
        $this->assertContains($fifteen->uuid, $ids, 'Range between 10 and 20 should find 15');
        $this->assertContains($twenty->uuid, $ids, 'Range between 10 and 20 should find 20');
        $this->assertNotContains($twentyFive->uuid, $ids, 'Range between 10 and 20 should not find 25');
    }

    public function testReturnsStatusIndicatorWithMinAndMax(): void
    {
        $this->subject->setState([ 'min' => 20, 'max' => '80' ]);
        $indicators = $this->subject->getStatusIndicators();
        $this->assertCount(1, $indicators);
        $this->assertEquals('20 - 80', $indicators[0]->getTitle());
    }

    public function testDoesNotReturnStatusIndicatorIfBoundsDefault(): void
    {
        $this->subject->setState([ 'min' => 0, 'max' => 100 ]);
        $this->assertCount(0, $this->subject->getStatusIndicators());
    }

    public function testIncludesUnitInStatusIndicator(): void
    {
        $this->subject->setState([ 'min' => 20, 'max' => '80' ]);
        $this->subject->setConfiguration([ 'unit' => 'meters' ]);
        $indicators = $this->subject->getStatusIndicators();
        $this->assertCount(1, $indicators);
        $this->assertEquals('20 - 80 meters', $indicators[0]->getTitle());
    }

    public function testStatusIndicatorIdUsedInRenderedOutput(): void
    {
        $this->subject->setState([ 'min' => 20, 'max' => '80' ]);
        $indicators = $this->subject->getStatusIndicators();
        $this->assertCount(1, $indicators);
        $this->assertContains($indicators[0]->getId(), $this->renderSubject());
    }

    public function providesWeirdRanges(): array
    {
        return [
            [ [ 'min' => 50, 'max' => 40 ], 0, 40 ],
            [ [ 'min' => -5, 'max' => 40 ], 0, 40 ],
            [ [ 'min' => 50, 'max' => 110 ], 50, 100 ],
            [ [ 'min' => -5, 'max' => 110 ], 0, 100 ],
            [ [ 'min' => 110, 'max' => 105 ], 0, 100 ],
        ];
    }

    public function provideInvalidState(): array
    {
        return [
            [ [ 'MIN' => 5, 'MAX' => 10 ] ],
            [ null ],
            [ [5,10] ],
            [ -1 ],
            [ 'lizard' ],
            [ [ PHP_INT_MAX ] ]
        ];
    }
}
