<?php declare(strict_types=1);

namespace Newland\NeosCommon\Tests\SignalSlot;


use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\ContentRepository\Domain\Repository\NodeDataRepository;
use Neos\ContentRepository\Domain\Repository\WorkspaceRepository;
use Neos\ContentRepository\Domain\Service\NodeTypeManager;
use Neos\Flow\Persistence\PersistenceManagerInterface;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosCommon\SignalSlot\NodeSlot;

class NodeSlotTest extends FunctionalTestCase
{
    /*
     * Enable persistence & forcibly destroy database after each test case:
     * - These tests assume that no previous nodes are existent.
     */
    protected static $testablePersistenceEnabled = true;
    protected $testableSecurityEnabled = true;

    /** @var ContentContext */
    private $context;

    /** @var Node */
    private $parentNode;

    /** @var Node */
    private $node;

    /** @var NodeSlot */
    private $subject;

    public function setUp()
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(NodeSlot::class);
        $this->context = new ContentContext('testing', new \DateTime(), [], [], false, false, false);
        $this->parentNode = $this->initializeNode('/test-site');
        $this->node = $this->initializeNode('/test-site/testing');
    }

    /**
     * @dataProvider getTitlesAndUris
     * @param string $title
     * @param string $uriSegment
     */
    public function testTransformsTitleIntoUriSegment(string $title, string $uriSegment): void
    {
        $this->node->setProperty('title', $title);
        $this->subject->nodePropertyChanged($this->node, 'title', '', $title);
        $this->assertEquals($uriSegment, $this->node->getProperty('uriPathSegment'));
    }

    /**
     * @dataProvider getTitlesAndUris
     * @depends testTransformsTitleIntoUriSegment
     * @param string $title
     * @param string $uriSegment
     */
    public function testEnsuresDuplicateUriPathSegmentsDoNotExist(string $title, string $uriSegment): void
    {
        // Set title for first node
        $this->node->setProperty('title', $title);
        $this->subject->nodePropertyChanged($this->node, 'title', '', $title);

        // Set title for second node: It now should have a `-1` suffix
        $secondNode = $this->initializeNode('/test-site/test2');
        $this->subject->nodePropertyChanged($secondNode, 'title', '', $title);
        $this->assertNotEquals($uriSegment, $secondNode->getProperty('uriPathSegment'));
    }

    public function getTitlesAndUris(): array
    {
        return [
            [ 'Foo ## (bar) baz', 'foo-bar-baz' ]
        ];
    }

    private function initializeNode(string $path): Node
    {
        /** @var WorkspaceRepository $workspaceRepository */
        $workspaceRepository = $this->objectManager->get(WorkspaceRepository::class);
        $workspace = $workspaceRepository->findOneByName('testing');
        if (!$workspace) {
            $workspace = new Workspace('testing');
            $workspaceRepository->add($workspace);
        }

        /** @var NodeTypeManager $nodeTypeManager */
        $nodeTypeManager = $this->objectManager->get(NodeTypeManager::class);
        $type = $nodeTypeManager->getNodeType('Neos.NodeTypes:Page');

        $data = new NodeData($path, $workspace);
        $data->setNodeType($type);

        $node = new Node($data, $this->context);
        $this->context->getFirstLevelNodeCache()->flush();
        $this->persistNode($node);
        return $node;
    }

    private function persistNode(Node $node): void
    {
        /** @var PersistenceManagerInterface $persistenceManager */
        $persistenceManager = $this->objectManager->get(PersistenceManagerInterface::class);

        /** @var NodeDataRepository $nodeDataRepository */
        $nodeDataRepository = $this->objectManager->get(NodeDataRepository::class);
        if ($persistenceManager->isNewObject($node->getNodeData())) {
            $nodeDataRepository->add($node->getNodeData());
        } else {
            $nodeDataRepository->update($node->getNodeData());
        }

        $persistenceManager->persistAll();
    }
}