<?php
namespace Newland\NeosCommon\Service;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\EntityManagerInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Routing\ObjectPathMapping;
use Neos\Flow\Mvc\Routing\ObjectPathMappingRepository;
use Neos\Flow\Persistence\PersistenceManagerInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

class ObjectPathMappingService
{
    /**
     * @var ObjectPathMappingRepository
     * @Flow\Inject()
     */
    protected $objectPathMappingRepository;

    /**
     * @var PersistenceManagerInterface
     * @Flow\Inject()
     */
    protected $persistenceManager;

    /**
     * @var EntityManagerInterface
     * @Flow\Inject()
     */
    protected $entityManager;

    /**
     * Flush all records from the object path mapping table for a certain object type.
     *
     * @param string $objectType
     */
    public function flushMappings(string $objectType, array $ids = null): void
    {
        $objectType = trim($objectType, '\\');

        if ($ids !== null) {
            $query = $this->objectPathMappingRepository->createQuery();
            $query->matching(
                $query->logicalAnd([
                    $query->equals('objectType', $objectType),
                    $query->in('identifier', $ids)
                ])
            );
            $records = $query->execute();
        } else {
            $records = $this->objectPathMappingRepository->findByObjectType($objectType);
        }

        foreach ($records as $record) {
            /** @var ObjectPathMapping $record */
            $this->objectPathMappingRepository->remove($record);
        }

        $this->persistenceManager->persistAll();
    }

    /**
     * Flush orphan records from the object path mapping table for a certain object type.
     */
    public function removeOrphans(string $objectType): void
    {
        $objectType = trim($objectType, '\\');

        $queryBuilder = $this->entityManager->createQueryBuilder();

        $result = $queryBuilder
            ->select('opm.identifier')
            ->from(ObjectPathMapping::class, 'opm')
            ->leftJoin(Article::class, 'a', 'WITH', 'a.Persistence_Object_Identifier = opm.identifier')
            ->where($queryBuilder->expr()->eq('opm.objectType', ':objectType'))
            ->andWhere('a.Persistence_Object_Identifier IS NULL')
            ->setParameter('objectType', $objectType)
            ->getQuery()
            ->getArrayResult();

        $ids = array_column($result, 'identifier');

        $this->flushMappings($objectType, $ids);
    }
}
