<?php
namespace Newland\NeosCommon\Service;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use ArrayAccess;
use Exception;
use Neos\Flow\Property\Exception\InvalidPropertyException;

class RecordService
{
    /**
     * @param object $record
     * @return string|bool|null
     */
    public function getProperty($record, string $property)
    {
        try {
            if (strpos($property, '.') !== false) {
                $propertyValue = $this->resolveDotNotation($record, $property);
            } else {
                $method = 'get' . ucfirst($property);
                $propertyValue = $record->$method();
            }
        } catch (Exception $e) {
            throw new InvalidPropertyException(
                'Property could not be resolved. Check the scenario configuration.'
            );
        }

        return $propertyValue;
    }

    /**
     * @param object $record
     * @param string $property
     * @return string|bool|null
     */
    public function resolveDotNotation($record, string $property)
    {
        try {
            $bits = explode('.', $property);
            $containerProperty = (string) array_shift($bits);
            $method = 'get' . ucfirst($containerProperty);
            $container = $record->$method();
            if ($container === null) {
                return null;
            }

            $childProperty = (string) array_shift($bits);
            if (is_numeric($childProperty)) {
                if (!(is_array($container) || $container instanceof ArrayAccess)
                    || !isset($container[$childProperty])
                ) {
                    throw new InvalidPropertyException('Invalid array offset.');
                }
                // if it's a collection of some sort, we get the element by index
                $container = $container[$childProperty];
                $childProperty = (string) array_shift($bits);
            }
            $method = $this->getMethod($container, $childProperty);
            return $container->$method();
        } catch (Exception $e) {
            return null;
        }
    }

    private function getMethod(object $container, string $childProperty): string
    {
        foreach ([ 'get', 'has', 'is' ] as $accessPrefix) {
            $methodName = $accessPrefix . ucfirst($childProperty);
            if (method_exists($container, $methodName)) {
                return $methodName;
            }
        }

        return 'get' . ucfirst($childProperty);
    }

    /**
     * @param object $record
     * @return bool|string|null
     */
    public function getPropertyWithFallback($record, array $properties)
    {
        foreach ($properties as $property) {
            $value = $this->getProperty($record, $property);
            if ($value) {
                return $value;
            }
        }
        return null;
    }
}
