<?php
namespace Newland\NeosCommon\Service;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;

/**
 * Shared handling of regions.
 */
abstract class AbstractRegionService
{
    /**
     * @var ConfigurationManager
     * @Flow\Inject()
     */
    protected $configurationManager;
    /**
     * @var array
     */
    protected $regionsAvailable;
    /**
     * @var array
     */
    protected $regionsEnabled;
    /**
     * @var array
     */
    private $regionsParsed = [];

    public function getZipsFromRegionKeys(array $regionsEnabled): array
    {
        return $this->collectZipsFromRegionKeys($regionsEnabled);
    }

    public function getAllEnabled(NodeInterface $node): array
    {
        $regionsEnabled = $this->getRegionsEnabledCached($node);
        return array_filter(
            $this->getRegionsAvailableCached(),
            function ($key) use ($regionsEnabled) {
                return in_array($key, $regionsEnabled, true);
            },
            ARRAY_FILTER_USE_KEY
        );
    }

    public function collectZipsForRegions(array $regions): array
    {
        $childZips = [ [] ];
        foreach ($regions as $region) {
            // validate configuration
            if (!is_array($region) || !array_key_exists('contains', $region)) {
                throw new InvalidConfigurationException('Region is not configured.', 1551107299);
            }

            // avoid loops
            if (array_key_exists($region['name'], $this->regionsParsed)) {
                continue;
            }
            $this->regionsParsed[$region['name']] = true;

            if ($region['contains'] === 'zips') {
                $childZips[] = $region['children'];
            } elseif ($region['contains'] === 'regions') {
                $childZips[] = $this->collectZipsFromRegionKeys($region['children']);
            }
        }

        return array_merge(... $childZips);
    }

    /**
     * @param array $regionKeys
     * @param bool $reset
     * @return array
     * @throws InvalidConfigurationException
     */
    public function collectZipsFromRegionKeys(array $regionKeys, $reset = false): array
    {
        if ($reset) {
            $this->reset();
        }

        return $this->collectZipsForRegions($this->getConcreteConfigForRegions($regionKeys));
    }

    /**
     * Fetches the available regions for all domains.
     * The returned array must have the following format:
     *
     * ```
     * {
     *     'region1': {
     *          'name': 'Foobar',
     *          'contains': 'zips',
     *          'children': [ '12345', '45667' ]
     *      },
     *     'combinedRegion': {
     *          'name': 'Foobar',
     *          'contains': 'regions',
     *          'children': [ 'region1', 'region2' ]
     *      },
     *      ...
     * }
     * ```
     *
     * @return array
     */
    abstract protected function getRegionsAvailable(): array;

    /**
     * Fetches the enabled regions for the current domain.
     *
     * @param NodeInterface $node
     * @return array
     */
    abstract protected function getRegionsEnabled(NodeInterface $node): array;

    protected function getRegionsAvailableCached(): array
    {
        if ($this->regionsAvailable === null) {
            $this->regionsAvailable = $this->getRegionsAvailable();
        }

        return $this->regionsAvailable;
    }

    protected function getRegionsEnabledCached(NodeInterface $node): array
    {
        if ($this->regionsEnabled === null) {
            $this->regionsEnabled = $this->getRegionsEnabled($node);
        }

        return $this->regionsEnabled;
    }

    protected function getConcreteConfigForRegions(array $regionKeys): array
    {
        $regions = [];

        $regionsAvailable = $this->getRegionsAvailable();
        foreach ($regionKeys as $regionKey) {
            if ($regionsAvailable[$regionKey] ?? false) {
                $regions[$regionKey] = $regionsAvailable[$regionKey];
            }
        }

        return $regions;
    }

    private function reset(): void
    {
        $this->regionsParsed = [];
    }
}
