<?php
namespace Newland\NeosCommon\Service;

use GuzzleHttp\Psr7\ServerRequest;
use GuzzleHttp\Psr7\Uri;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\ActionResponse;
use Neos\Flow\Mvc\Controller\Arguments;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Mvc\Routing\UriBuilder;
use Neos\Neos\Domain\Model\Domain;
use Neos\Neos\Domain\Model\Site;
use Neos\Neos\Domain\Service\ContentContext;
use Psr\Http\Message\UriInterface;

/**
 * @Flow\Scope("singleton")
 */
class ControllerContextFactory
{

    /**
     * Initializes a new fake & independent controller context.
     * The controller context that is being created mimics a controller context that is available
     * during a HTTP Request but can also be built during CLI commands.
     *
     * NOTE: During the initialization the global environment variables are changed since neos internals
     *       assume that these are set. If you create multiple contexts be sure that you create them immediately
     *       before using them instead of creating them in bulk.
     *
     * @example
     * // Correct: Initialize as you need
     * foreach ($nodes as $node) {
     *      $this->articleCache->generateAndStoreInCache(
     *          $this->createQuery($node),
     *          $this->contextFactory->initializeFakeControllerContext($node)
     *      );
     * }
     *
     * @example
     * // Incorrect: Initialize before you need
     * $contexts = array_map(
     *      function($node) { return $this->contextFactory->initializeFakeControllerContext($node) },
     *      $nodes
     * );
     *
     * // ...
     *
     * @param NodeInterface $node
     * @param array $environment
     * @return ControllerContext
     */
    public function initializeFakeControllerContext(NodeInterface $node, array $environment = []): ControllerContext
    {
        $environment = $this->applyDefaultEnvironment($environment);
        $this->applyEnvironmentVariables($environment);

        $uriBuilder = new UriBuilder();
        $uriBuilder->setRequest($this->initializeFakeActionRequest($node, $environment));

        return new ControllerContext(
            $uriBuilder->getRequest(),
            new ActionResponse(),
            new Arguments([]),
            $uriBuilder
        );
    }

    public function initializeFakeActionRequest(NodeInterface $node, array $env = []): ActionRequest
    {
        $baseUri = $this->primaryDomainUri($node) ?? new Uri();
        $httpRequest = new ServerRequest('GET', $baseUri, [], null, '1.1', $env);

        $request = ActionRequest::fromHttpRequest($httpRequest);
        $request->setArgument('__node', $node);

        return $request;
    }

    private function applyDefaultEnvironment(array $environment): array
    {
        if (!array_key_exists('FLOW_REWRITEURLS', $environment)) {
            $environment['FLOW_REWRITEURLS'] = 1;
        }
        if (!array_key_exists('SCRIPT_NAME', $environment)) {
            $environment['SCRIPT_NAME'] = '/';
        }
        return $environment;
    }


    private function primaryDomainUri(NodeInterface $node): ?UriInterface
    {
        $context = $node->getContext();
        if (!($context instanceof ContentContext)) {
            return null;
        }

        /** @var Site|null $site */
        $site = $context->getCurrentSite();
        if ($site === null) {
            return null;
        }

        /** @var Domain|null $domain */
        $domain = $site->getPrimaryDomain();
        if ($domain === null) {
            return null;
        }

        return $this->domainToUri($domain);
    }


    private function domainToUri(Domain $domain): UriInterface
    {
        $uri = new Uri('/');
        $uri = $uri->withHost($domain->getHostname());

        $scheme = $domain->getScheme();
        if (!$scheme) {
            $scheme = $this->getCurrentProtocol();
        }
        $uri = $uri->withScheme($scheme);

        $port = $domain->getPort();
        if ($port) {
            $uri = $uri->withPort($port);
        }

        return $uri;
    }

    private function applyEnvironmentVariables(array $environment)
    {
        foreach ($environment as $name => $value) {
            putenv($name . '=' . $value);
        }
    }

    /**
     * Extract the protocol from the current request
     *
     * @return string
     */
    private function getCurrentProtocol(): string
    {
        return ($this->requestIsOverHttps()) ? 'https' : 'http';
    }

    private function requestIsOverHttps(): bool
    {
        if (!empty($_SERVER['HTTPS'])) {
            return $_SERVER['HTTPS'] !== 'off';
        }
        if (!empty($_SERVER['SERVER_PORT'])) {
            return ((int) $_SERVER['SERVER_PORT']) === 443;
        }
        return false;
    }
}
