/**
 * Data Loader similar to data loaders in the Neos Core.
 * The following methods must be implemented:
 * - search(options, searchTerm): Array
 * - resolveValue(options, identifier): mixed
 *
 * @see Packages/Application/Neos.Neos.Ui/packages/neos-ui/src/manifest.dataloaders.js
 * @implements DataLoaderInterface
 */
import { stripMultilineIndention } from '../utility';

export default class LinkHandlerDataLoader {

    /**
     * @public
     * @param {{}} options
     * @param {string} searchTerm
     * @returns {Promise<DataLoaderInterface~Link[]>}
     */
    search (options, searchTerm) {
        if (options.nodeTypes.indexOf('Neos.Neos:Document') === -1) {
            console.warn(stripMultilineIndention(`
                Linking configuration does not include Neos.Neos:Document which means that this is not a
                wildcard search field that searches for all record types (all record types extend Neos.Neos:Document).
                Therefor Linkhandler is not active.
            `));
            return Promise.resolve([]);
        }

        const language = options.contextForNodeLinking.dimensions.language[0];
        return this.debouncedSearchPromise()
            .then(() => this.executeRequest({ language, searchTerm }));
    }

    /**
     * @public
     * @param {{}} options
     * @param {string} identifier
     * @returns {Promise<DataLoaderInterface~Value[]>}
     */
    resolveValue (options, identifier) {
        return this.executeRequest({ 'recordIdentifiers[0]': identifier })
            .then(results => {
                results.forEach(res => res.icon = 'database');
                return results;
            });
    }

    /**
     * @private
     */
    executeRequest (query = {}) {
        const queryString = [];
        for (const key of Object.keys(query)) {
            queryString.push(`${key}=${encodeURIComponent(query[ key ])}`);
        }

        return fetch(`/neos/records?${queryString.join('&')}`)
            .then(response => {
                if (response.status === 404) {
                    console.error(stripMultilineIndention(`
                         The \`/neos/records\` API-Endpoint that should be handled by Newland.NeosCommon was not found.
                         This is most likely because the project Routes.yaml does not include Newland.NeosCommon.
                         To fix this issue make sure you include the followin in your projects Routes.yaml:
                         
                             - name: 'Common'
                               uriPattern: '<CommonSubroutes>'
                               subRoutes:
                                 'CommonSubroutes':
                                   package: 'Newland.NeosCommon'
                         
                         Please refer to the README.md file of Newland.NeosCommon in order to see all stepse
                         required to setup the LinkHandler component.
                    `));
                    return [];
                }

                return response.json()
            }).then(result => {
                if (result.error) {
                    console.error(
                        'An error occurred while calling the `/neos/records` API-Endpoint handled by Newland.NeosCommon: \n\n' +
                        result.error + '\n\n' +
                        result.message
                    );
                    if (result.trace) {
                        console.table(result.trace);
                    }
                    return [];
                }

                return result;
            });
    }

    /**
     * @private
     * @param {int} ms
     * @returns {Promise<void>}
     */
    debouncedSearchPromise(ms = 300) {
        return new Promise(resolve => {
            if (this._debounceTimer) {
                window.clearTimeout(this._debounceTimer);
            }
            this._debounceTimer = window.setTimeout(resolve, ms);
        })
    }

}
