<?php declare(strict_types=1);
namespace Newland\NeosCommon\Controller;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ActionController;
use Newland\NeosCommon\LinkHandler\LinkHandlerFactory;
use Newland\NeosCommon\LinkHandler\Domain\Model\Record;

/**
 * Rudimentary REST service for nodes
 *
 * @Flow\Scope("singleton")
 */
class RecordController extends ActionController
{
    /**
     * @Flow\Inject
     * @var LinkHandlerFactory
     */
    protected $linkHandlerFactory;

    public function callActionMethod(): void
    {
        try {
            parent::callActionMethod();
        } catch (\Exception $e) {
            $this->response->setContent(
                (string) json_encode(
                    [
                        'error' => get_class($e),
                        'message' => $e->getMessage(),
                        'trace' => $e->getTrace(),
                    ]
                )
            );
            $this->response->setHttpHeader('Content-Type', 'application/json');
            $this->response->setStatusCode(500);
        }
    }


    /**
     * Shows a list of nodes
     */
    public function indexAction(
        string $searchTerm = '',
        array $recordIdentifiers = [],
        array $recordTypes = [],
        string $language = null,
    ): string {
        /** @var Record[][] $records */
        if ($recordIdentifiers === []) {
            $records = $this->search($searchTerm, $recordTypes ?: $this->linkHandlerFactory->allTypes(), $language);
        } else {
            $records = $this->findByIdentifiers($recordIdentifiers);
        }

        $formatted = [];
        foreach ($records as $type => $typeRecords) {
            foreach ($typeRecords as $record) {
                $formatted[] = [
                    'dataType' => 'Newland.NeosCommon:Linkhandler',
                    'loaderUri' => 'record://' . $type . ':' . $record->getId(),
                    'label' => $record->getTitle(),
                    'identifier' => $record->getId(),
                ];
            }
        }

        $this->response->setHttpHeader('Content-Type', 'application/json');

        return json_encode($formatted);
    }


    /**
     * @return Record[][]
     */
    private function search(string $searchTerm, array $recordTypes, string $language = null): array
    {
        $results = [];
        foreach ($recordTypes as $type) {
            $results[$type] = $this->linkHandlerFactory->get($type, $language)->findBySearchTerm($searchTerm);
        }
        return $results;
    }

    /**
     * @return Record[][]
     */
    private function findByIdentifiers(array $recordIdentifiers): array
    {
        $identifiersByType = [];
        foreach ($recordIdentifiers as $identifier) {
            if (preg_match('/([a-zA-Z0-9\-]+):([a-z0-9\-]+)/', $identifier, $matches)) {
                [ , $recordType, $identifier ] = $matches;
                $identifiersByType[$recordType] = $identifiersByType[$recordType] ?? [];
                $identifiersByType[$recordType][] = $identifier;
            }
        }

        $results = [];
        foreach ($identifiersByType as $type => $identifiers) {
            $results[$type] = [];
            $handler = $this->linkHandlerFactory->get((string) $type);

            foreach ($identifiers as $identifier) {
                $result = $handler->findByIdentifier($identifier);
                if ($result) {
                    $results[$type][] = $result;
                }
            }
        }
        return $results;
    }
}
