<?php declare(strict_types=1);

namespace Newland\NeosCommon\Tests\Unit\LinkHandler;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Neos\Domain\Service\ContentContext;
use Neos\Neos\Domain\Service\ContentContextFactory;
use Newland\NeosCommon\LinkHandler\Domain\Model\Record;
use Newland\NeosCommon\LinkHandler\LinkHandlerFactory;
use Newland\NeosCommon\LinkHandler\LinkingService;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\NeosCommon\Tests\Unit\LinkHandler\Mock\LinkHandlerMock;
use Newland\NeosTestingHelpers\InteractsWithNodes;

class LinkingServiceTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;
    use InteractsWithNodes;

    /** @var LinkingService */
    private $subject;

    /** @var Node */
    private $node;

    /** @var ContentContext */
    private $context;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(LinkingService::class);
        $this->node = $this->initializeNode('/sites/node-foobar');
        $this->context = $this->objectManager->get(ControllerContextFactory::class)
            ->initializeFakeControllerContext($this->node);
    }

    /**
     * @dataProvider provideNodeUrisForWorkaround
     * @param string $input
     * @param string $expectedOutput
     */
    public function testAppliesNodeUriWorkaroundCorrectly(string $input, string $expectedOutput): void
    {
        $this->assertEquals(
            $expectedOutput,
            $this->subject->nodeUriWorkaround($input)
        );
    }

    public function provideNodeUrisForWorkaround(): array
    {
        return [
            // Short move node://record -> record://
            [ 'node://record:article:foobar', 'record://article:foobar' ],
            [
                'node://record:article:5b054363-17f7-4eef-88b8-c6e273551dcc',
                'record://article:5b054363-17f7-4eef-88b8-c6e273551dcc',
            ],

            // Should not change anything else
            [
                'record://article:96dcfc9c-362b-48c6-9d3e-b3019cf0b6a5',
                'record://article:96dcfc9c-362b-48c6-9d3e-b3019cf0b6a5',
            ],
            [ 'node://4b32beeb-a7f5-4256-a9df-197dd0a69be4', 'node://4b32beeb-a7f5-4256-a9df-197dd0a69be4' ],
            [ 'asset://3d2fe4fc-3678-4cac-9bbe-3a5dcd8c106c', 'asset://3d2fe4fc-3678-4cac-9bbe-3a5dcd8c106c' ],
        ];
    }

    /**
     * @dataProvider provideLinkhandlerData
     * @param string $recordType
     * @param string $generatedLink
     * @param string $uri
     * @param string $expected
     */
    public function testUsesConfiguredLinkhandlerToGenerateLinkIfUriMatches(string $recordType, string $generatedLink, string $uri, ?string $expected): void
    {
        $this->injectLinkhandlerConfiguration(
            [
                'recordTypes' => [
                    $recordType => [
                        'handler' => LinkHandlerMock::class,
                        'linkToGenerate' => $generatedLink,
                        'recordToFind' => new Record(),
                    ],
                ],
            ]
        );

        $this->assertEquals(
            $expected,
            $this->subject->resolveRecordUri($uri, $this->node, $this->context, true)
        );
    }


    /**
     * @dataProvider provideLinkhandlerData
     * @param string $recordType
     * @param string $generatedLink
     * @param string $uri
     * @param string $expected
     */
    public function testDoesNotGenerateLinkIfTheConfiguredLinkhandlerDoesNotFindARecord(string $recordType, string $generatedLink, string $uri, ?string $expected): void
    {
        $this->injectLinkhandlerConfiguration(
            [
                'recordTypes' => [
                    $recordType => [
                        'handler' => LinkHandlerMock::class,
                        'linkToGenerate' => $generatedLink,
                        'recordToFind' => null, // No record found
                    ],
                ],
            ]
        );

        $this->assertEquals(
            null,
            $this->subject->resolveRecordUri($uri, $this->node, $this->context, true)
        );
    }


    public function provideLinkhandlerData(): array
    {
        return [
            [ 'test', 'https://foobar.com', 'record://test:foo', 'https://foobar.com' ],
            [ 'test', 'https://foobar.com', 'record://lalala:foo', null ],
        ];
    }

    private function injectLinkhandlerConfiguration(array $configuration): void
    {
        $factory = $this->objectManager->get(LinkHandlerFactory::class);
        $this->inject($factory, 'configuration', $configuration);
        $this->inject($this->subject, 'linkHandlerFactory', $factory);
    }

}