<?php declare(strict_types=1);

namespace Newland\NeosCommon\Tests\ViewHelpers;

use Neos\Flow\Tests\FunctionalTestCase;
use Neos\FluidAdaptor\View\StandaloneView;

class PaginationViewHelperTest extends FunctionalTestCase
{

    public function testRendersNothingIfOnlyOnePageAvailable(): void
    {
        $this->assertEmpty($this->renderViewHelper(1, 1));
    }

    /** @dataProvider providePaginationItems */
    public function testRendersPaginationItemsCorrectly(int $currentPage, int $totalPages, array $items): void
    {
        $rendered = $this->renderViewHelper($currentPage, $totalPages);
        $this->assertNotEmpty($items);
        foreach ($items as $item) {
            $this->assertStringContainsString(sprintf('data-toubiz-pagination-page="%d"', $item), $rendered);
        }
    }

    public function providePaginationItems(): array
    {
        return [
            'first couple of pages are rendered regularly' => [ 3, 55, [ 1, 2, 3, 4, 5 ] ],
            'always renders first page' => [ 30, 55, [ 1 ] ],
            'always renders last page' => [ 30, 55, [ 55 ] ],
        ];
    }

    /** @dataProvider provideRunningPagination */
    public function testAlwaysRendersFirstPage(int $currentPage, int $totalPages): void
    {
        $this->assertStringContainsString(
            'data-toubiz-pagination-page="1"',
            $this->renderViewHelper($currentPage, $totalPages)
        );
    }

    /** @dataProvider provideRunningPagination */
    public function testAlwaysRendersLastPage(int $currentPage, int $totalPages): void
    {
        $this->assertStringContainsString(
            sprintf('data-toubiz-pagination-page="%d"', 1),
            $this->renderViewHelper($currentPage, $totalPages)
        );
    }

    /** Provides each page for a pagination of 20 pages. */
    public function provideRunningPagination(): array
    {
        $total = 20;
        $pages = [];
        for ($i = 1; $i < $total; $i++) {
            $pages[] = [ $i, $total ];
        }
        return $pages;
    }

    public function testRendersCorrectlyIfCurrentPageNumberToLow(): void
    {
        $rendered = $this->renderViewHelper(-2, 10);
        $this->assertStringNotContainsString('data-toubiz-pagination-page="-2"', $rendered);
    }

    public function testRendersCorrectlyIfCurrentPageNumberToHigh(): void
    {
        $rendered = $this->renderViewHelper(15, 10);
        $this->assertStringNotContainsString('data-toubiz-pagination-page="15"', $rendered);
    }

    public function testDoesNotRenderIfNegativeTotalPages(): void
    {
        $this->assertEmpty($this->renderViewHelper(1, -2));
    }

    private function renderViewHelper(int $current, int $total): string
    {
        $renderer = new StandaloneView();
        $renderer->setTemplateSource(
            sprintf(
                '
                    {namespace test=Newland\NeosCommon\ViewHelpers}
                    <test:pagination currentPage="%d" totalPageCount="%d" />
                ',
                $current,
                $total
            )
        );
        return trim($renderer->render());
    }

}
