<?php
namespace Newland\NeosCommon\Command;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cli\CommandController;

/**
 * Abstract command controller.
 *
 * Base for all command controllers.
 *
 * @Flow\Scope("singleton")
 */
abstract class AbstractCommandController extends CommandController
{
    const EVENT_PROGRESS = 'progress';
    const EVENT_START = 'start';
    const EVENT_END = 'end';

    /**
     * @var \Closure[][]
     */
    protected $eventListeners = [];

    protected function showProgressOnCommandLine()
    {
        $this->on(static::EVENT_START, function (string $type) {
            $this->output->progressStart();
            $this->output->output(' ' . $type . "\r");
        });
        $this->on(static::EVENT_PROGRESS, function (string $type, int $processed) {
            $this->output->progressSet($processed);
            $this->output->output(' ' . $type . "\r");
        });
        $this->on(static::EVENT_END, function (string $type) {
            $this->output->progressFinish();
            $this->output->outputLine(' ' . $type);
        });
    }


    protected function on(string $event, \Closure $listener)
    {
        if (!array_key_exists($event, $this->eventListeners)) {
            $this->eventListeners[$event] = [];
        }

        $this->eventListeners[$event][] = $listener;
    }


    /**
     * @Flow\Signal()
     * @param string $type
     */
    protected function emitStart(string $type)
    {
        $this->callListeners('start', $type);
    }

    /**
     * @Flow\Signal()
     * @param string $type
     */
    protected function emitEnd(string $type)
    {
        $this->callListeners('end', $type);
    }

    /**
     * @Flow\Signal()
     * @param string $type
     * @param int $processedElements
     */
    protected function emitProgress(string $type, int $processedElements)
    {
        $this->callListeners('progress', $type, $processedElements);
    }

    private function callListeners(string $event, ...$payload)
    {
        if (!array_key_exists($event, $this->eventListeners)) {
            return;
        }

        foreach ($this->eventListeners[$event] as $listener) {
            $listener(...$payload);
        }
    }
}
