<?php
namespace Newland\NeosCommon\Service;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\PersistenceManagerInterface;
use Newland\NeosCommon\Domain\Repository\RecordRepository;
use Newland\NeosCommon\Exceptions\ConfigurationError;

/**
 * Find nodes based on a fulltext search
 *
 * @Flow\Scope("singleton")
 */
class RecordSearchService implements RecordSearchServiceInterface
{
    /**
     * @Flow\Inject
     * @var PersistenceManagerInterface
     */
    protected $persistenceManager;
    /**
     * @var array
     * @Flow\InjectConfiguration(path="linkhandler")
     */
    protected $config;
    /**
     * @var RecordRepository
     * @Flow\Inject
     */
    protected $recordRepository;

    /**
     * @param array $recordIdentifiers
     * @return array|null
     */
    public function findByIdentifiers(array $recordIdentifiers)
    {
        $results = [];
        foreach ($recordIdentifiers as $identifier) {
            $record = $this->findByIdentifier($identifier);

            if ($record) {
                if (array_key_exists($record['type'], $results) === false) {
                    $results[$record['type']] = [];
                }
            }

            $results[$record['type']][] = $record['record'];
        }

        return $results;
    }

    /**
     * @param string $searchTerm
     * @param array $recordTypes
     * @return array|null
     */
    public function findBySearchFields(string $searchTerm, array $recordTypes)
    {
        $results = [];

        $recordTypesConfig = $this->getRecordTypesConfig($recordTypes);
        if (empty($recordTypesConfig)) {
            throw new ConfigurationError('
                No record type configuration found in Newland.NeosCommon.linkhandler.recordTypes.
                Please refer to the README.md file to learn how to setup record types for linkhandling.
            ');
        }

        foreach ($recordTypesConfig as $recordType => $config) {
            $results[$recordType] = $this->recordRepository->findBySearchField(
                $searchTerm,
                $this->getQueryParameters($config, (string) $recordType)
            );
        }

        return $results;
    }

    /**
     * @param array $recordTypes
     * @return array
     */
    protected function getRecordTypesConfig(array $recordTypes): array
    {
        $recordTypesConfig = [];

        if (empty($recordTypes)) {
            $recordTypes = array_keys($this->config['recordTypes'] ?? []);
        }

        foreach ($recordTypes as $recordType) {
            $recordTypesConfig[$recordType] = $this->getRecordTypeConfig($recordType);
        }

        return $recordTypesConfig;
    }

    /**
     * @param string $recordType
     * @return array
     */
    protected function getRecordTypeConfig($recordType): array
    {
        return $this->config['recordTypes'][$recordType];
    }

    protected function getQueryParameters(array $config, string $recordType): array
    {
        return [
            'alias' => $recordType,
            'objectType' => $config['objectType'],
            'searchField' => $config['searchField'],
            'filter' => $config['filter'] ?? [],
        ];
    }

    /**
     * @param string $recordIdentifier
     * @return array|null
     */
    protected function findByIdentifier(string $recordIdentifier)
    {
        $result = null;
        if (preg_match('/([a-zA-Z0-9\-]+):([a-z0-9\-]+)/', $recordIdentifier, $matches)) {
            $recordType = $matches[1];
            $identifier = $matches[2];
            $recordTypeConfig = $this->getRecordTypeConfig($recordType);
            $queryParameters = $this->getQueryParameters($recordTypeConfig, $recordType);

            $record = $this->recordRepository->findByRecordIdentifier($identifier, $queryParameters);
            if ($record) {
                $result = [
                    'type' => $recordType,
                    'record' => $record,
                ];
            }
        }

        return $result;
    }
}
