<?php
namespace Newland\NeosCommon\Controller;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Http\Headers;
use Neos\Flow\Mvc\Controller\ActionController;
use Newland\NeosCommon\Domain\Repository\RecordRepository;
use Newland\NeosCommon\Service\RecordSearchServiceInterface;

/**
 * Rudimentary REST service for nodes
 *
 * @Flow\Scope("singleton")
 */
class RecordController extends ActionController
{
    /**
     * @Flow\Inject
     * @var RecordSearchServiceInterface
     */
    protected $recordSearchService;
    /**
     * @var RecordRepository
     * @Flow\Inject
     */
    protected $recordRepository;

    public function callActionMethod()
    {
        try {
            parent::callActionMethod();
        } catch (\Exception $e) {
            $this->response->setContent(
                (string) json_encode(
                    [
                        'error' => get_class($e),
                        'message' => $e->getMessage(),
                        'trace' => $e->getTrace(),
                    ]
                )
            );
            $this->response->setHeader('Content-Type', 'application/json');
            $this->response->setStatus(500);
        }
    }

    /**
     * Shows a list of nodes
     *
     * @param string $searchTerm An optional search term used for filtering the list of nodes
     * @param array $recordIdentifiers An optional list of node identifiers
     * @param array $recordTypes
     * @return false|string
     */
    public function indexAction($searchTerm = '', array $recordIdentifiers = [], array $recordTypes = [])
    {
        if ($recordIdentifiers === []) {
            $recordTypeResults = $this->recordSearchService->findBySearchFields($searchTerm, $recordTypes);
        } else {
            $recordTypeResults = $this->recordSearchService->findByIdentifiers($recordIdentifiers);
        }

        $formatted = [];
        foreach ($recordTypeResults ?: [] as $type => $results) {
            foreach ($results as $result) {
                $formatted[] = [
                    'dataType' => 'Newland.NeosCommon:Linkhandler',
                    'loaderUri' => 'record://' . $type . ':' . $result['id'],
                    'label' => $result['title'],
                    'identifier' => $result['id'],
                ];
            }
        }

        /** @var Headers $headers */
        $headers = $this->response->getHeaders();
        $headers->set('Content-Type', 'application/json');

        return json_encode($formatted);
    }
}
