<?php
namespace Newland\NeosCommon\Service;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Http\Uri;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Property\PropertyMapper;
use Neos\Media\Domain\Model\AssetInterface;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Psr\Log\LogLevel;

/**
 * A service for creating URIs pointing to nodes, assets and records.
 *
 * The target node can be provided as string or as a Node object; if not specified
 * at all, the generated URI will refer to the current document node inside the Fusion context.
 *
 * When specifying the ``node`` argument as string, the following conventions apply:
 *
 * *``node`` starts with ``/``:*
 * The given path is an absolute node path and is treated as such.
 * Example: ``/sites/acmecom/home/about/us``
 *
 * *``node`` does not start with ``/``:*
 * The given path is treated as a path relative to the current node.
 * Examples: given that the current node is ``/sites/acmecom/products/``,
 * ``stapler`` results in ``/sites/acmecom/products/stapler``,
 * ``../about`` results in ``/sites/acmecom/about/``,
 * ``./neos/info`` results in ``/sites/acmecom/products/neos/info``.
 *
 * *``node`` starts with a tilde character (``~``):*
 * The given path is treated as a path relative to the current site node.
 * Example: given that the current node is ``/sites/acmecom/products/``,
 * ``~/about/us`` results in ``/sites/acmecom/about/us``,
 * ``~`` results in ``/sites/acmecom``.
 *
 * @Flow\Scope("singleton")
 */
class LinkingService extends \Neos\Neos\Service\LinkingService
{
    /**
     * Pattern to match supported URIs.
     *
     * @var string
     */
    const PATTERN_SUPPORTED_URIS = '/(node|asset|record):\/\/([a-z0-9\-]{8}\-[a-z0-9\-]{4}\-[a-z0-9\-]{4}\-[a-z0-9\-]{4}\-[a-z0-9\-]{12}|([a-zA-Z\-]+):([a-z0-9\-]+))/';
    /**
     * @var PropertyMapper
     * @Flow\Inject
     */
    protected $propertyMapper;
    /**
     * @var NodeService
     * @Flow\Inject
     */
    protected $nodeService;
    /**
     * @var Node[]
     */
    protected $nodeCache = [];
    /**
     * @var array
     * @Flow\InjectConfiguration(path="linkhandler")
     */
    protected $configuration = [];

    /**
     * Resolves a given record:// URI to a "normal" HTTP(S) URI for the detail page displaying the addressed record.
     *
     * @param string $uri
     * @param NodeInterface $contextNode
     * @param ControllerContext $controllerContext
     * @param bool $absolute
     * @return null|string
     */
    public function resolveRecordUri(
        string $uri,
        NodeInterface $contextNode,
        ControllerContext $controllerContext,
        $absolute = false
    ) {
        $detailPage = $this->convertUriToObject($uri, $contextNode);

        $nodeUri = null;

        if ($detailPage !== null) {
            $arguments = $this->getDetailPageArguments($uri);
            $nodeUri = $this->createNodeUri($controllerContext, $detailPage, null, null, $absolute, $arguments);
        } else {
            $this->systemLogger->log(
                LogLevel::ALERT,
                sprintf('Could not resolve "%s" to an existing detail page.', $uri)
            );
        }

        return $nodeUri;
    }

    /**
     * Return the object the URI addresses or NULL.
     *
     * @param string|Uri $uri
     * @param NodeInterface $contextNode
     * @return NodeInterface|AssetInterface|object|NULL
     */
    public function convertUriToObject($uri, NodeInterface $contextNode = null)
    {
        if ($uri instanceof Uri) {
            $uri = (string) $uri;
        }

        if (preg_match(self::PATTERN_SUPPORTED_URIS, $uri, $matches) === 1) {
            switch ($matches[1]) {
                case 'node':
                    if ($contextNode === null) {
                        throw new \RuntimeException(
                            'node:// URI conversion requires a context node to be passed',
                            1409734235
                        );
                    };

                    return $contextNode->getContext()->getNodeByIdentifier($matches[2]);
                case 'asset':
                    return $this->assetRepository->findByIdentifier($matches[2]);
                case 'record':
                    return $this->findDetailPageForNodeType($matches[3], $contextNode);
            }
        }

        return null;
    }

    /**
     * @param string $nodeType
     * @param NodeInterface $contextNode
     * @return Node|null
     * @throws \Neos\Eel\Exception
     */
    protected function findDetailPageForNodeType(string $nodeType, NodeInterface $contextNode)
    {
        $detailPage = '';
        try {
            $detailPage = $this->nodeForRecord(
                $nodeType,
                $contextNode->getContext()->getCurrentSiteNode()
            );
        } catch (\Exception $e) {
        }

        return $detailPage;
    }

    /**
     * @param string $nodeType
     * @param NodeInterface $site
     * @return null|Node
     */
    protected function nodeForRecord(string $nodeType, NodeInterface $site)
    {
        $key = $site->getPath() . ':' . $nodeType;
        if (!array_key_exists($key, $this->nodeCache)) {
            $selector = $this->configuration['recordTypes'][$nodeType]['eelPluginSelector'];
            $nodes = (new FlowQuery([ $site ]))
                ->find($selector)
                ->parents('[instanceof Neos.NodeTypes:Page]')
                ->get();
            $node = $this->nodeService->getFirstAvailableNode($nodes);
            $this->nodeCache[$key] = $node;
        }

        return $this->nodeCache[$key];
    }

    /**
     * @param string $uri
     * @return mixed
     */
    private function getDetailPageArguments(string $uri)
    {
        preg_match(self::PATTERN_SUPPORTED_URIS, $uri, $matches);

        $arguments = $this->configuration['recordTypes'][$matches[3]]['arguments'];

        $namespace = array_keys($arguments)[0];
        $parameterName = $arguments[$namespace]['parameterName'];
        $arguments[$namespace][$parameterName] = $this->getRecord($uri);
        unset($arguments[$namespace]['parameterName']);

        return $arguments;
    }

    private function getRecord(string $uri)
    {
        preg_match(self::PATTERN_SUPPORTED_URIS, $uri, $matches);

        $recordType = $matches[3];
        $id = $matches[4];

        return $this->propertyMapper->convert(
            $id,
            $this->configuration['recordTypes'][$recordType]['objectType']
        );
    }

}
