<?php
namespace Newland\NeosCommon\Eel\Helper;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\Eel\ProtectedContextAwareInterface;
use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Property\Exception\InvalidPropertyException;
use Neos\Flow\Property\PropertyMapper;
use Newland\NeosCommon\Exceptions\ArgumentsNotFoundException;
use Newland\NeosCommon\Models\MetaData;

/**
 * Abstract Eel Helper to handle properties from single records displayed in detail view.
 */
abstract class ObjectPropertyHelper implements ProtectedContextAwareInterface
{
    /**
     * @var ConfigurationManager
     * @Flow\Inject
     */
    protected $configurationManager;
    /**
     * @var PropertyMapper
     * @Flow\Inject
     */
    protected $propertyMapper;
    /**
     * @var MetaData
     * @Flow\Inject
     */
    protected $metaData;
    /**
     * @var Node
     */
    protected $node;
    /**
     * @var ActionRequest
     */
    protected $request;
    /**
     * @var array
     */
    protected $pluginParameterSets = [];
    /**
     * @var array
     */
    protected $pluginParameters = [];
    /**
     * @var array
     */
    protected $scenarios = [];
    /**
     * @var array
     */
    protected $activeScenario = [];
    /**
     * @var string
     */
    protected $value = '';
    /**
     * @var string
     */
    protected $websiteTitle = '';
    /**
     * @var object|null
     */
    protected $record;
    /**
     * @var bool
     */
    protected $override;

    /**
     * @param string $methodName
     *
     * @return boolean
     */
    public function allowsCallOfMethod($methodName): bool
    {
        return true;
    }

    /**
     * Get the configuration
     */
    protected function getConfiguration()
    {
        $defaultConfiguration = $this->configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.NeosCommon.meta'
        );

        try {
            $this->scenarios = $defaultConfiguration['scenarios'];
            $this->override = $defaultConfiguration['override'];
        } catch (\Exception $e) {
            // Graceful exit if no configuration is found, because "override" will not be set.
        }
    }

    /**
     * Set value from requested record
     */
    protected function setValueFromRequestedRecord()
    {
        $this->parseConfiguration();
        $this->setMetaDataFromScenario();
    }

    /**
     * Parse plugin parameters and scenarios for a match
     */
    protected function parseConfiguration()
    {
        foreach ($this->request->getPluginArguments() as $pluginParameters) {
            foreach ($this->scenarios as $scenario) {
                if ($this->pluginParametersMatchWithScenario($pluginParameters, $scenario)) {
                    $this->pluginParameters = $pluginParameters;
                    $this->activeScenario = $scenario;
                    return;
                }
            }
        }
    }

    /**
     * Check for a match
     *
     * @param array $pluginParameters
     * @param array $scenario
     * @return bool
     */
    protected function pluginParametersMatchWithScenario(array $pluginParameters, array $scenario): bool
    {
        try {
            if (
                $pluginParameters['@controller'] === strtolower($scenario['controller'])
                && $pluginParameters['@package'] === strtolower($scenario['package'])
                && $pluginParameters['@action'] === strtolower($scenario['action'])
                && array_key_exists($scenario['parameter'], $pluginParameters)
            ) {
                return true;
            }

        } catch (\Exception $exception) {
            throw new ArgumentsNotFoundException(
                'No matching set of arguments found. Possible misconfiguration in scenario configuration.'
            );
        }

        return false;
    }

    /**
     * Set the detail record as object
     */
    protected function setMetaDataFromScenario()
    {
        try {
            $record = $this->propertyMapper->convert(
                $this->pluginParameters[$this->activeScenario['parameter']]['__identity'],
                $this->activeScenario['objectType']
            );
            if (isset($this->activeScenario['propertyMapping']['title'])) {
                $this->metaData->setTitle(
                    $this->getProperty($record, $this->activeScenario['propertyMapping']['title'])
                );
            }
            if (isset($this->activeScenario['propertyMapping']['description'])) {
                $this->metaData->setDescription(
                    $this->getProperty($record, $this->activeScenario['propertyMapping']['description'])
                );
            }
            if (isset($this->activeScenario['propertyMapping']['image'])) {
                $this->metaData->setImage(
                    $this->getProperty($record, $this->activeScenario['propertyMapping']['image'])
                );
            }
            $this->metaData->setUrl((string) $this->request->getHttpRequest()->getUri());
            if (isset($this->activeScenario['ogType'])) {
                $this->metaData->setType($this->activeScenario['ogType']);
            }
        } catch (\Exception $e) {
        }
    }

    public function getMetaData()
    {
        return $this->metaData;
    }

    private function getProperty($record, string $property): string
    {
        try {
            if (strpos($property, '.') !== false) {
                $propertyValue = $this->resolveDotNotation($record, $property);
            } else {
                $method = 'get' . ucfirst($property);
                $propertyValue = $record->$method();
            }
        } catch (\Exception $e) {
            throw new InvalidPropertyException(
                'Property could not be resolved. Check the scenario configuration.'
            );
        }

        return (string) $propertyValue;
    }

    /**
     * @param object $record
     * @param string $property
     * @return string
     */
    private function resolveDotNotation($record, string $property)
    {
        $value = '';
        try {
            $bits = explode('.', $property);
            $containerProperty = array_shift($bits);
            $method = 'get' . ucfirst($containerProperty);
            $container = $record->$method();
            $childProperty = array_shift($bits);
            if (is_numeric($childProperty)) {
                if (!(is_array($container) || $container instanceof \ArrayAccess) ||
                    !isset($container[$childProperty])) {
                    throw new InvalidPropertyException('Invalid array offset.');
                }
                // if it's a collection of some sort, we get the element by index
                $container = $container[$childProperty];
                $childProperty = array_shift($bits);
            }
            $method = 'get' . ucfirst($childProperty);
            $value = $container->$method();
        } catch (\Exception $e) {
        }

        return $value;
    }
}
