<?php
namespace Newland\NeosCommon\Domain\Repository;

/*
 * This file is part of the "neos-common" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;

/**
 * Class RecordRepository
 *
 * @Flow\Scope("singleton")
 */
class RecordRepository
{
    /**
     * @var \Doctrine\Common\Persistence\ObjectManager
     * @Flow\Inject
     */
    protected $entityManager;

    /**
     * @param string|array $searchTerm
     * @param array $searchFields
     * @return array
     */
    public function findBySearchFields($searchTerm, array $searchFields)
    {
        if (\count($searchFields) === 0) {
            throw new InvalidConfigurationException('No search fields configured.', 1544779983);
        }

        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->entityManager->createQueryBuilder();
        foreach ($searchFields as $searchField) {
            $queryBuilder
                ->addSelect($searchField['alias'] . '.Persistence_Object_Identifier AS id')
                ->addSelect($searchField['alias'] . '.' . $searchField['searchField'] . ' AS title')
                ->from($searchField['objectType'], $searchField['alias'])
                ->where(
                    $queryBuilder->expr()->like(
                        $searchField['alias'] . '.' . $searchField['searchField'],
                        ':searchTerm'
                    )
                )
                ->setParameter('searchTerm', '%' . $searchTerm . '%');
        }

        return $queryBuilder->orderBy('title')
            ->getQuery()->execute(null, Query::HYDRATE_ARRAY);
    }

    public function findBySearchField(string $searchTerm, array $searchField)
    {
        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->entityManager->createQueryBuilder();
        $queryBuilder
            ->select($searchField['alias'] . '.Persistence_Object_Identifier AS id')
            ->addSelect($searchField['alias'] . '.' . $searchField['searchField'] . ' AS title')
            ->from($searchField['objectType'], $searchField['alias'])
            ->where(
                $queryBuilder->expr()->like(
                    $searchField['alias'] . '.' . $searchField['searchField'],
                    ':searchTerm'
                )
            )
            ->setParameter('searchTerm', '%' . $searchTerm . '%');

        foreach ($searchField['filter'] as $field => $value) {
            $queryBuilder
                ->andWhere(
                    $queryBuilder->expr()->like(
                        $searchField['alias'] . '.' . $field,
                        ':' . $field
                    )
                )
                ->setParameter($field, $value);
        }

        return $queryBuilder->orderBy('title')->getQuery()->execute(null, Query::HYDRATE_ARRAY);

    }

    public function findByRecordIdentifier(string $identifier, array $queryParameters)
    {
        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->entityManager->createQueryBuilder();
        $queryBuilder
            ->select($queryParameters['alias'] . '.Persistence_Object_Identifier AS id')
            ->addSelect($queryParameters['alias'] . '.' . $queryParameters['searchField'] . ' AS title')
            ->from($queryParameters['objectType'], $queryParameters['alias'])
            ->where(
                $queryBuilder->expr()->like(
                    $queryParameters['alias'] . '.Persistence_Object_Identifier',
                    ':identifier'
                )
            )
            ->setParameter('identifier', $identifier)
            ->setMaxResults(1);

        $result = $queryBuilder->getQuery()->execute();
        return empty($result) ? null : $result[0];
    }
}
