<?php
namespace Newland\NeosCommon\Service;

use Neos\Flow\Annotations as Flow;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Http\Request;
use Neos\Flow\Http\Response;
use Neos\Flow\Http\Uri;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\Controller\Arguments;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Mvc\Routing\UriBuilder;
use Neos\Neos\Domain\Model\Domain;

/**
 * @Flow\Scope("singleton")
 */
class ControllerContextFactory
{

    /**
     * Initializes a new fake & independent controller context.
     * The controller context that is being created mimics a controller context that is available
     * during a HTTP Request but can also be built during CLI commands.
     *
     * NOTE: During the initialization the global environment variables are changed since neos internals
     *       assume that these are set. If you create multiple contexts be sure that you create them immediately
     *       before using them instead of creating them in bulk.
     *
     * @example
     * // Correct: Initialize as you need
     * foreach ($nodes as $node) {
     *      $this->articleCache->generateAndStoreInCache(
     *          $this->createQuery($node),
     *          $this->contextFactory->initializeFakeControllerContext($node)
     *      );
     * }
     *
     * @example
     * // Incorrect: Initialize before you need
     * $contexts = array_map(
     *      function($node) { return $this->contextFactory->initializeFakeControllerContext($node) },
     *      $nodes
     * );
     *
     * // ...
     *
     * @param Node $node
     * @param array $environment
     * @return ControllerContext
     */
    public function initializeFakeControllerContext(Node $node, array $environment = []): ControllerContext
    {
        $environment = $this->applyDefaultEnvironment($environment);
        $this->applyEnvironmentVariables($environment);
        $uriBuilder = $this->initializeUriBuilder($node, $environment);

        return new ControllerContext(
            $uriBuilder->getRequest(),
            new Response(),
            new Arguments([]),
            $uriBuilder
        );
    }

    private function applyDefaultEnvironment(array $environment): array
    {
        if (!array_key_exists('FLOW_REWRITEURLS', $environment)) {
            $environment['FLOW_REWRITEURLS'] = 1;
        }
        if (!array_key_exists('SCRIPT_NAME', $environment)) {
            $environment['SCRIPT_NAME'] = '/';
        }
        return $environment;
    }

    private function initializeUriBuilder(Node $node, array $env): UriBuilder
    {
        $httpRequest = new Request([], [], [], $env);
        $httpRequest->setContent(null);

        $domain = $node->getContext()->getCurrentSite()->getPrimaryDomain();
        if ($domain) {
            $httpRequest->setBaseUri($this->domainToUri($domain));
        }

        $request = new ActionRequest($httpRequest);
        $request->setArgument('__node', $node);

        $uriBuilder = new UriBuilder();
        $uriBuilder->setRequest($request);

        return $uriBuilder;
    }


    private function domainToUri(Domain $domain): Uri
    {
        $uri = new Uri('/');
        $uri = $uri->withHost($domain->getHostname());

        $scheme = $domain->getScheme();
        if ($scheme) {
            $uri = $uri->withScheme($scheme);
        }

        $port = $domain->getPort();
        if ($port) {
            $uri = $uri->withPort($port);
        }

        return $uri;
    }

    private function applyEnvironmentVariables(array $environment)
    {
        foreach ($environment as $name => $value) {
            putenv($name . '=' . $value);
        }
    }
}
