<?php
namespace Newland\NeosCommon\Service;

use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;

/**
 * Shared handling of regions.
 */
abstract class AbstractRegionService
{

    /**
     * Fetches the available regions.
     * The returned array must have the following format:
     *
     * ```
     * {
     *     'region1': {
     *          'name': 'Foobar',
     *          'contains': 'zips',
     *          'children': [ '12345', '45667' ]
     *      },
     *     'combinedRegion': {
     *          'name': 'Foobar',
     *          'contains': 'regions',
     *          'children': [ 'region1', 'region2' ]
     *      },
     *      ...
     * }
     * ```
     *
     * @param ConfigurationManager $configurationManager
     * @return array
     */
    abstract protected function regionsAvailable(ConfigurationManager $configurationManager): array;

    /**
     * Fetches the enabled regions.
     * The returned array must contain keys in the `regionsAvailable` array.
     *
     * @param ConfigurationManager $configurationManager
     * @return array
     */
    abstract protected function regionsEnabled(ConfigurationManager $configurationManager): array;

    /**
     * @var array
     */
    protected $regionsAvailable;

    public function injectRegionsAvailable(ConfigurationManager $configurationManager)
    {
        $this->regionsAvailable  = $this->regionsAvailable($configurationManager);
    }

    /**
     * @var array
     */
    protected $regionsEnabled;

    public function injectRegionsEnabled(ConfigurationManager $configurationManager)
    {
        $this->regionsEnabled  = $this->regionsEnabled($configurationManager);
    }

    /**
     * @var array
     */
    private $regionsParsed = [];

    public function getZipsFromRegionKeys(array $regionsEnabled): array
    {
        return $this->collectZipsFromRegionKeys($regionsEnabled);
    }

    public function getAllEnabled(): array
    {
        return array_filter(
            $this->regionsAvailable,
            function ($key) {
                return in_array($key, $this->regionsEnabled);
            },
            ARRAY_FILTER_USE_KEY
        );
    }

    public function collectZipsForRegions(array $regions, array $zips = []): array
    {
        foreach ($regions as $region) {
            // validate configuration
            if (!is_array($region) || !array_key_exists('contains', $region)) {
                throw new InvalidConfigurationException('Region is not configured.', 1551107299);
            }

            // avoid loops
            if (array_key_exists($region['name'], $this->regionsParsed)) {
                continue;
            }
            $this->regionsParsed[$region['name']] = true;

            $childZips = [];
            if ($region['contains'] === 'zips') {
                $childZips = $region['children'];
            } elseif ($region['contains'] === 'regions') {
                $childZips = $this->collectZipsFromRegionKeys($region['children']);
            }

            $zips = array_merge($zips, $childZips);
        }

        return $zips;
    }

    /**
     * @param array $regionKeys
     * @param bool $reset
     * @return array
     * @throws InvalidConfigurationException
     */
    public function collectZipsFromRegionKeys(array $regionKeys, $reset = false): array
    {
        if ($reset) {
            $this->reset();
        }

        return $this->collectZipsForRegions(
            $this->getConcreteConfigForRegions($regionKeys)
        );
    }

    protected function getConcreteConfigForRegions(array $regionKeys): array
    {
        $regions = [];

        foreach ($regionKeys as $regionKey) {
            if (array_key_exists($regionKey, $this->regionsAvailable)) {
                $regions[$regionKey] = $this->regionsAvailable[$regionKey];
            }
        }

        return $regions;
    }

    private function reset()
    {
        $this->regionsParsed = [];
    }
}
