<?php
namespace Newland\Maileon\Neos\Service;

/*
 * This file is part of the "maileon-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;

/**
 * API service class for accessing the Maileon API.
 */
class ApiService
{
    /**
     * @var string Base URI for requests.
     */
    const BASE_URI = 'https://api.maileon.com/1.0/';

    /**
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * Creates a new contact.
     *
     * Default setting is to do nothing if the contact already exists.
     *
     * @see http://dev.maileon.com/api/rest-api-1-0/contacts/create-contact-json/
     */
    public function createContact($email, $fields)
    {
        if (empty($fields['custom_fields'])) {
            unset($fields['custom_fields']);
        }

        $response = $this->httpClient()->post(
            'contacts/email/' . urlencode($email),
            [
                'headers' => [
                    'Content-Type' => 'application/vnd.maileon.api+json'
                ],
                'json' => $fields
            ]
        );

        if ($response->getStatusCode() == 200) {
            return json_decode((string)$response->getBody(), true);
        } else {
            return null;
        }
    }

    /**
     * Gets custom fields defined in the Maileon account.
     *
     * @return array
     */
    public function getCustomFields()
    {
        $response = $this->httpClient()->get('contacts/fields/custom');
        if ($response->getStatusCode() == 200) {
            $response = new \SimpleXMLElement((string)$response->getBody());
            $data = ((array)$response)['custom_field'];

            $fields = [];
            foreach ($data as $item) {
                $fields[] = (string)$item->name;
            }

            sort($fields);
            return $fields;
        } else {
            return [];
        }
    }

    /**
     * Gets target groups defined in the Maileon account.
     *
     * @return array
     */
    public function getTargetGroups()
    {
        $response = $this->httpClient()->get('targetgroups');
        if ($response->getStatusCode() == 200) {
            $response = new \SimpleXMLElement((string)$response->getBody());
            $data = (array)$response;

            $groups = [];
            foreach ($data['targetgroup'] as $item) {
                $groups[] = [
                    'id' => (integer)$item->id,
                    'name' => (string)$item->name
                ];
            }

            usort(
                $groups,
                function($a, $b) {
                    if (strtolower($a['name']) == strtolower($b['name'])) {
                        return 0;
                    }
                    return (strtolower($a['name']) > strtolower($b['name'])) ? +1 : -1;
                }
            );
            return $groups;
        } else {
            return [];
        }
    }

    /**
     * Builds a guzzle http client.
     *
     * @return \GuzzleHttp\Client
     */
    protected function httpClient()
    {
        return new \GuzzleHttp\Client([
            'auth' => [
                $this->configuration['apiKey'], null, 'Basic'
            ],
            'base_uri' => static::BASE_URI
        ]);
    }
}
