<?php
namespace Newland\Maileon\Neos\Finisher;

/*
 * This file is part of the "maileon-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Form\Core\Model\AbstractFinisher;
use Neos\Form\Exception\FinisherException;
use Newland\Maileon\Neos\Service\ApiService;

/**
 * Finisher for form.
 */
class MaileonFinisher extends AbstractFinisher
{
    /**
     * @Flow\Inject
     * @var ApiService
     */
    protected $apiService;

    /**
     * @Flow\Inject
     * @var Neos\ContentRepository\Domain\Service\ContextFactoryInterface
     */
    protected $contextFactory;

    /**
     * @var array
     */
    protected $fields = [
        'standard_fields' => [],
        'custom_fields' => []
    ];

    /**
     * Concrete finisher method.
     *
     * @return void
     * @throws FinisherException
     */
    protected function executeInternal()
    {
        if (!$this->shouldSubmit()) {
            return;
        }

        foreach ($this->finisherContext->getFormValues() as $id => $value) {
            $node = $this->findFormNode($id);
            $property = $node->getProperty('maileonField');
            if (!empty($property)) {
                $this->addToFields($property, $value);
            }
        }

        if (
            array_key_exists('EMAIL', $this->fields['standard_fields'])
            && !empty($this->fields['standard_fields']['EMAIL'])
        ) {
            $this->apiService->createContact(
                $this->fields['standard_fields']['EMAIL'],
                $this->fields
            );
        }
    }

    /**
     * Finds the node for the given id.
     *
     * @param string $id
     * @return \Neos\ContentRepository\Domain\Model\NodeInterface
     */
    protected function findFormNode($id)
    {
        $context = $this->contextFactory->create([
            'workspaceName' => 'live',
            'currentDateTime' => new \Neos\Flow\Utility\Now(),
            'dimensions' => array(),
            'invisibleContentShown' => false,
            'removedContentShown' => false,
            'inaccessibleContentShown' => false
        ]);

        return (new FlowQuery([$context->getCurrentSiteNode()]))
            ->find('#' . $id)
            ->get(0);
    }

    /**
     * @var string $property
     * @var string $value
     * @return void
     */
    protected function addToFields($property, $value)
    {
        list($scope, $name) = explode('.', $property);
        $this->fields[$scope][$name] = $value;
    }

    /**
     * Checks if the given form data should be submitted to maileon.
     *
     * @return bool
     */
    protected function shouldSubmit()
    {
        $submissionFilter = $this->parseOption('submissionFilter');
        if ($submissionFilter == 'always') {
            return true;
        }

        if ($submissionFilter == 'activeSubmissionField') {
            foreach ($this->finisherContext->getFormValues() as $id => $value) {
                $node = $this->findFormNode($id);
                $property = $node->getProperty('submissionFilterField');

                if (!empty($property)) {
                    if ((bool)$value === true) {
                        /* Submit if one of the fields is configured to be true.
                         * This approach is used because you may have two
                         * checkboxes with a target group mapping (which defines
                         * to which contact listing the user subscribes) and
                         * the user should be subscribed to either one or both
                         * or none; If no submission field must be false, one
                         * can only subscribe to _all_ or _none_.
                         */
                        return true;
                    }
                }
            }

            return false;
        }
    }
}
