<?php
namespace Newland\Maileon\Neos\Service;

/*
 * This file is part of the "maileon-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Client;
use Neos\Flow\Annotations as Flow;
use SimpleXMLElement;

/**
 * API service class for accessing the Maileon API.
 */
class ApiService
{
    /**
     * @var string Base URI for requests.
     */
    const BASE_URI = 'https://api.maileon.com/1.0/';

    /**
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * Creates a new contact.
     *
     * Default setting is to do nothing if the contact already exists.
     *
     * @param string $email
     * @param array $fields
     * @see http://dev.maileon.com/api/rest-api-1-0/contacts/create-contact-json/
     */
    public function createContact($email, array $fields)
    {
        if (empty($fields['custom_fields'])) {
            unset($fields['custom_fields']);
        }

        $path = sprintf('contacts/email/%s?doi=true&doiplus=false', urlencode($email));

        $response = $this->httpClient()->post(
            $path,
            [
                'headers' => [
                    'Content-Type' => 'application/vnd.maileon.api+json',
                ],
                'json' => $fields,
            ]
        );

        if ($response->getStatusCode() === 200) {
            return json_decode((string) $response->getBody(), true);
        }

        return null;
    }

    /**
     * Gets custom fields defined in the Maileon account.
     */
    public function getCustomFields(): array
    {
        $response = $this->httpClient()->get('contacts/fields/custom');
        if ($response->getStatusCode() === 200) {
            $response = new SimpleXMLElement((string) $response->getBody());
            $data = ((array) $response)['custom_field'];

            $fields = [];
            foreach ($data as $item) {
                $fields[] = (string) $item->name;
            }

            sort($fields);
            return $fields;
        }

        return [];
    }

    /**
     * Builds a guzzle http client.
     */
    protected function httpClient(): Client
    {
        return new Client(
            [
                'auth' => [
                    $this->configuration['apiKey'],
                    null,
                    'Basic',
                ],
                'base_uri' => static::BASE_URI,
            ]
        );
    }
}
