<?php
namespace Newland\Maileon\Neos\Finisher;

/*
 * This file is part of the "maileon-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\ArrayPropertyCollection;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\ContentRepository\Domain\Service\ContextFactoryInterface;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Utility\Now;
use Neos\Form\Core\Model\AbstractFinisher;
use Newland\Maileon\Neos\Service\ApiService;

/**
 * Finisher for form.
 *
 * @Flow\Scope("singleton")
 */
class MaileonFinisher extends AbstractFinisher
{
    /**
     * @Flow\Inject
     * @var ApiService
     */
    protected $apiService;

    /**
     * @Flow\Inject
     * @var ContextFactoryInterface
     */
    protected $contextFactory;

    /**
     * @var array
     */
    protected $fields = [
        'standard_fields' => [],
        'custom_fields' => [],
    ];

    /**
     * Concrete finisher method.
     */
    protected function executeInternal(): void
    {
        if (!$this->shouldSubmit()) {
            return;
        }

        foreach ($this->finisherContext->getFormValues() as $id => $value) {
            $node = $this->findFormNode($id);
            if ($node === null) {
                continue;
            }

            $property = $node->getProperty('maileonField');
            if (empty($property)) {
                continue;
            }

            $this->addToFields($property, $value);
        }

        $this->emitAfterFieldsProcessed($this->fields);

        if ($this->fields['standard_fields']['EMAIL'] ?? false) {
            $this->apiService->createContact(
                $this->fields['standard_fields']['EMAIL'],
                $this->fields
            );
        }
    }

    /**
     * Finds the node for the given id.
     *
     * @param string $id
     * @return NodeInterface|null
     */
    protected function findFormNode($id)
    {
        $context = $this->contextFactory->create(
            [
                'workspaceName' => 'live',
                'currentDateTime' => new Now(),
                'dimensions' => [],
                'invisibleContentShown' => false,
                'removedContentShown' => false,
                'inaccessibleContentShown' => false,
            ]
        );

        return (new FlowQuery([ $context->getCurrentSiteNode() ]))
            ->find('#' . $id)
            ->get(0);
    }

    /**
     * @var string $value
     */
    protected function addToFields(string $property, $value): void
    {
        // Return on empty value to not fill array with nonsense.
        if (empty($value)) {
            return;
        }

        [ $scope, $name ] = explode('.', $property);
        $this->fields[$scope][$name] = $value;
    }

    /**
     * Checks if the given form data should be submitted to maileon.
     */
    protected function shouldSubmit(): bool
    {
        $submissionFilter = $this->parseOption('submissionFilter');
        if ($submissionFilter === 'always') {
            return true;
        }

        if ($submissionFilter === 'activeSubmissionField') {
            foreach ($this->finisherContext->getFormValues() as $id => $value) {
                $node = $this->findFormNode($id);
                if ($node === null) {
                    continue;
                }
                $property = $node->getProperty('submissionFilterField');

                if (!empty($property) && (bool) $value === true) {
                    /*
                     * Submit if one of the fields is configured to be true.
                     * This approach is used because you may have two
                     * checkboxes with a target group mapping (which defines
                     * to which contact listing the user subscribes) and
                     * the user should be subscribed to either one or both
                     * or none; If no submission field must be false, one
                     * can only subscribe to _all_ or _none_.
                     */
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Signal stub to modify processed fields before API submission.
     *
     * @param array &$fields
     * @return void
     * @Flow\Signal
     */
    protected function emitAfterFieldsProcessed(array &$fields)
    {
    }

    /**
     * Wrapper around the original parseOption() to fetch values from
     * the given ArrayPropertyCollection if they're not found by the
     * original method.
     *
     * This is a workaround as this has stopped working in between unknown
     * neos/flow/form versions.
     *
     * @param string $optionName
     * @return mixed
     */
    protected function parseOption($optionName)
    {
        $option = parent::parseOption($optionName);
        if (!empty($option)) {
            return $option;
        }

        if (is_a($this->options[0], ArrayPropertyCollection::CLASS)
            && ($this->options[0][$optionName] ?? false)) {
            return $this->options[0][$optionName];
        }
    }
}
