<?php declare(strict_types=1);

/*
 * Copyright (c) 2022, land in sicht AG
 *
 * www.land-in-sicht.de - All rights reserved.
 *
 * This is proprietary software. Unauthorized copying
 * of this file, via any medium, is strictly prohibited.
 */

namespace Newland\GpsFileParsing\Tests\Model;

use Newland\GpsFileParsing\Model\Point;
use Newland\GpsFileParsing\Model\Track;
use PHPUnit\Framework\TestCase;

class TrackTest extends TestCase
{
    public function testShouldBeAbleToMakeCalculationsFromPointArray(): void
    {
        $track = new Track();

        $track->addPoint(new Point(7.630552053, 47.59724322, 281.0));
        $track->addPoint(new Point(7.630004882, 47.59774965, 280.0));
        $track->addPoint(new Point(7.629060745, 47.59861059, 281.0));

        self::assertInstanceOf(Point::class, $track->getHighestPoint());
        self::assertInstanceOf(Point::class, $track->getLowestPoint());
        self::assertIsFloat($track->getTrackDistanceKilometers());
        self::assertIsFloat($track->getDescentMeters());
        self::assertIsFloat($track->getClimbMeters());
    }

    public function testFindsMinimumElevation(): void
    {
        $track = new Track();
        $track->addPoint(new Point(0.0, 0.0, 1.0));
        $track->addPoint(new Point(0.0, 0.0, 2.0));
        $track->addPoint(new Point(0.0, 0.0, 3.0));
        self::assertSame(1.0, $track->getLowestPoint()->getElevation());
    }

    public function testFindsMaximumElevation(): void
    {
        $track = new Track();
        $track->addPoint(new Point(0.0, 0.0, 1.0));
        $track->addPoint(new Point(0.0, 0.0, 2.0));
        $track->addPoint(new Point(0.0, 0.0, 3.0));
        self::assertSame(3.0, $track->getHighestPoint()->getElevation());
    }

    public function testFindsDescent(): void
    {
        $track = new Track();
        foreach ([ 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 5.0, 4.0, 3.0 ] as $elevation) {
            $track->addPoint(new Point(0.0, 0.0, $elevation));
        }
        self::assertSame(3.0, $track->getDescentMeters());
    }

    public function testFindsAscent(): void
    {
        $track = new Track();
        foreach ([ 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 5.0, 4.0, 3.0 ] as $elevation) {
            $track->addPoint(new Point(0.0, 0.0, $elevation));
        }
        self::assertSame(5.0, $track->getClimbMeters());
    }

    /** @dataProvider provideDistance */
    public function testCalculatesDistances(
        array $pointCoordinates,
        float $distance,
        float $leewayMultiplier = 0.01
    ): void {
        $track = new Track();
        foreach ($pointCoordinates as $coordinates) {
            $track->addPoint(new Point(...$coordinates));
        }

        $distanceMin = $distance - $distance * $leewayMultiplier;
        $distanceMax = $distance + $distance * $leewayMultiplier;
        self::assertGreaterThanOrEqual($distanceMin, $track->getTrackDistanceKilometers());
        self::assertLessThanOrEqual($distanceMax, $track->getTrackDistanceKilometers());
    }

    public function provideDistance(): array
    {
        return [
            [ [ [ 47.8475324, 7.6989516 ], [ 47.9809116, 7.818731 ] ], 17.31 ],
            [ [ [ 48.2884795, 7.8066703 ], [ 47.9424012, 7.7545737 ] ], 38.68 ],
            [
                [
                    [ 48.2884795, 7.8066703 ],
                    [ 47.9809116, 7.818731 ],
                    [ 47.9424012, 7.7545737 ],
                    [ 47.8475324, 7.6989516 ],
                ],
                34.21 + 6.42 + 11.33,
            ],
        ];
    }
}
