<?php declare(strict_types=1);

namespace Newland\GpsFileParsing\Tests\Model;

use Newland\GpsFileParsing\Model\Point;
use PHPUnit\Framework\TestCase;

class PointTest extends TestCase
{
    /** @dataProvider provideElevations */
    public function testCanFigureOutIfOnePointIsHigherThanAnother(float $elevationA, float $elevationB, bool $isHigher, bool $isLower): void
    {
        $pointA = new Point(0.0, 0.0, $elevationA);
        $pointB = new Point(0.0, 0.0, $elevationB);

        if ($isHigher) {
            $this->assertTrue($pointA->isHigherThan($pointB), sprintf('Point A [%s] must be higher than point B [%s]', $elevationA, $elevationB));
        } else {
            $this->assertFalse($pointA->isHigherThan($pointB), sprintf('Point A [%s] must not be higher than point B [%s]', $elevationA, $elevationB));
        }
    }

    /** @dataProvider provideElevations */
    public function testCanFigureOutIfOnePointIsLowerThanAnother(float $elevationA, float $elevationB, bool $isHigher, bool $isLower): void
    {
        $pointA = new Point(0.0, 0.0, $elevationA);
        $pointB = new Point(0.0, 0.0, $elevationB);

        if ($isLower) {
            $this->assertTrue($pointA->isLowerThan($pointB), sprintf('Point A [%s] must be lower than point B [%s]', $elevationA, $elevationB));
        } else {
            $this->assertFalse($pointA->isLowerThan($pointB), sprintf('Point A [%s] must not be lower than point B [%s]', $elevationA, $elevationB));
        }
    }

    /** @dataProvider provideDistance */
    public function testCanCalculateDistanceBetween2Points(array $coordinatesA, array $coordinatesB, float $distance, float $leewayMultiplier = 0.01): void
    {
        $pointA = new Point(...$coordinatesA);
        $pointB = new Point(...$coordinatesB);

        $distanceMin = $distance - $distance * $leewayMultiplier;
        $distanceMax = $distance + $distance * $leewayMultiplier;

        $aToB = $pointA->distance($pointB);
        $this->assertGreaterThanOrEqual($distanceMin, $aToB);
        $this->assertLessThanOrEqual($distanceMax, $aToB);

        $bToA = $pointB->distance($pointA);
        $this->assertGreaterThanOrEqual($distanceMin, $bToA);
        $this->assertLessThanOrEqual($distanceMax, $bToA);
    }

    public function provideElevations(): array
    {
        return [
            [ 1.0, 2.0, false, true ],
            [ 1.0, 1.0, false, false ],
            [ 2.0, 1.0, true, false ],
        ];
    }

    public function provideDistance(): array
    {
        return [
            [ [ 47.8475324, 7.6989516 ], [ 47.9809116, 7.818731 ], 17.31 ],
            [ [ 48.2884795, 7.8066703 ], [ 47.9424012, 7.7545737 ], 38.68 ],
        ];
    }
}
