<?php declare(strict_types=1);

namespace Newland\GpsFileParsing\Model;

class Track
{
    /** @var Point[] */
    protected $points = [];

    /** @return Point[] */
    public function getPoints(): array
    {
        return $this->points;
    }

    public function addPoint(Point $point): void
    {
        $this->points[] = $point;
    }

    public function isEmpty(): bool
    {
        return empty($this->points);
    }

    public function mapPoints(callable $block): array
    {
        return array_map($block, $this->points);
    }

    public function getHighestPoint(): ?Point
    {
        return array_reduce(
            $this->points,
            function (?Point $carry, Point $current) {
                if ($carry === null || $current->isHigherThan($carry)) {
                    return $current;
                }
                return $carry;
            }
        );
    }

    public function getLowestPoint(): ?Point
    {
        return array_reduce(
            $this->points,
            function (?Point $carry, Point $current) {
                if ($carry === null || $current->isLowerThan($carry)) {
                    return $current;
                }
                return $carry;
            }
        );
    }

    public function getDescentMeters(): float
    {
        $totalDescent = 0;

        foreach ($this->points as $i => $currentPoint) {
            $nextPoint = $this->points[$i + 1] ?? null;
            if ($nextPoint === null) {
                break;
            }

            if ($nextPoint->isLowerThan($currentPoint)) {
                $totalDescent += $currentPoint->getElevation() - $nextPoint->getElevation();
            }
        }

        return $totalDescent;
    }

    public function getClimbMeters(): float
    {
        $totalClimb = 0;

        foreach ($this->points as $i => $currentPoint) {
            $nextPoint = $this->points[$i + 1] ?? null;
            if ($nextPoint === null) {
                break;
            }

            if ($nextPoint->isHigherThan($currentPoint)) {
                $totalClimb += $nextPoint->getElevation() - $currentPoint->getElevation();
            }
        }

        return $totalClimb;
    }

    public function getTrackDistanceKilometers(): float
    {
        $totalDistance = 0;

        // To obtain a precise total track distance, we need to calculate the distance between each point.
        // If only the first and last point are taken into account,
        // any turns and variations from a straight line in-between will not be taken into account.
        foreach ($this->points as $i => $point) {
            // When we reach the last point in the array, we are done
            if (array_key_exists($i + 1, $this->points)) {
                $totalDistance += $point->distance($this->points[$i + 1]);
            }
        }

        return $totalDistance;
    }
}
