<?php declare(strict_types=1);

namespace Newland\GpsFileParsing\Parser;

use Newland\GpsFileParsing\Exception\Exception as GpsFileParsingException;
use Newland\GpsFileParsing\Helper\XmlFileReader;
use Newland\GpsFileParsing\Model\Point;
use Newland\GpsFileParsing\Model\Track;

class GpxParser implements Parser
{
    /** @var XmlFileReader */
    private $reader;

    public function __construct(XmlFileReader $reader)
    {
        $this->reader = $reader;
    }

    public function extractTrack(string $file, bool $silentFail = true): Track
    {
        $track = new Track();

        try {
            $data = $this->reader->read($file);
        } catch (GpsFileParsingException $e) {
            if ($silentFail) {
                return $track;
            }

            throw $e;
        }

        $this->parseTrackSegments($data, $track);

        if ($track->isEmpty()) {
            $this->parseRoutePoints($data, $track);
        }

        return $track;
    }

    private function parseTrackSegments(array $data, Track $track): void
    {
        // incoming data can either be
        // { 'trkseg': { 'trkpt': [ ... ] } }
        // or
        // { 'trkseg': [ { 'trkpt': [ ... ] }, { 'trkpt': [ ... ] } ] }
        $singleSegmentPoints = $data['trk']['trkseg']['trkpt'] ?? null;
        if ($singleSegmentPoints) {
            $segments = [ [ 'trkpt' => $singleSegmentPoints ] ];
        } else {
            $segments = $data['trk']['trkseg'] ?? [];
        }

        foreach ($segments as $segment) {
            foreach ($segment['trkpt'] as $trackPoint) {
                $elevation = array_key_exists('ele', $trackPoint) ? ((float) $trackPoint['ele']) : 0.0;
                $point = new Point(
                    (float) $trackPoint['@attributes']['lat'],
                    (float) $trackPoint['@attributes']['lon'],
                    $elevation
                );
                $track->addPoint($point);
            }
        }
    }

    private function parseRoutePoints(array $data, Track $track): void
    {
        foreach ($data['rte']['rtept'] ?? [] as $trackPoint) {
            $elevation = array_key_exists('ele', $trackPoint) ? ((float) $trackPoint['ele']) : 0.0;
            $point = new Point(
                (float) $trackPoint['@attributes']['lat'],
                (float) $trackPoint['@attributes']['lon'],
                $elevation
            );
            $track->addPoint($point);
        }
    }
}
