<?php declare(strict_types=1);


namespace Newland\ToubizApi\Tests\Unit\Utility;


use Newland\Toubiz\Api\Utility\ArrayUtility;
use PHPUnit\Framework\TestCase;

class ArrayUtilityTest extends TestCase
{

    /**
     * @dataProvider associativeArraysDataProvider
     * @param array $arrayToTest
     * @param bool $isAssociative
     */
    public function testDetectsAssociativeArrays(array $arrayToTest, bool $isAssociative): void
    {
        $this->assertEquals($isAssociative, ArrayUtility::isAssociative($arrayToTest));
    }

    public function associativeArraysDataProvider(): array
    {
        return [
            [ [ 'a' => 'b' ], true ],
            [ [ 1, 2, 3 ], false ],
            [ [ 1, 2, 'foo' => 'bar' ], true ]
        ];
    }

    /**
     * @dataProvider arrayValuesByPathDataProvider
     * @param array $array
     * @param string|array $path
     * @param string|array $expectedResult
     */
    public function testGetsArrayValuesByPath(array $array, $path, $expectedResult): void
    {
        $this->assertEquals($expectedResult, ArrayUtility::arrayGet($array, $path));
    }

    public function arrayValuesByPathDataProvider(): array
    {
        return [
            [
                [ 'foo' => [ 'bar' => 'baz' ] ],
                [ 'foo', 'bar' ],
                'baz'
            ],

            [
                [ 'foo' => [ 'bar' => 'baz' ] ],
                'foo',
                [ 'bar' => 'baz' ]
            ],

            [
                [ 'foo' => [ 'bar' => 'baz' ] ],
                [ 'path', 'does', 'not', 'exist' ],
                null
            ],
        ];
    }

    public function testExplodesTrimmedStrings(): void
    {
        $this->assertEquals([ 'foo', 'bar', 'baz' ], ArrayUtility::trimExplode(',', 'foo, bar   ,    baz'));
    }

    /**
     * @dataProvider arrayOfPrimitivesDataProvider
     * @param array $array
     * @param bool $isArrayOfPrimitives
     */
    public function testDetectsArrayOfPrimitives(array $array, bool $isArrayOfPrimitives): void
    {
        $this->assertEquals($isArrayOfPrimitives, ArrayUtility::isArrayOfPrimitives($array));
    }

    public function arrayOfPrimitivesDataProvider(): array
    {
        return [
            [ [ 1, 2, 3 ], true ],
            [ [ 'foo', 'bar' ], true ],
            [ [ 1, 'foo' ], true ],
            [ [ true, false ], true ],
            [ [ 'multi' => [ 'dimensional' => 'array' ] ], false ]
        ];
    }

    /**
     * @dataProvider twoDimensionalArrayOfPrimitivesDataProvider
     * @param array $array
     * @param bool $isPrimitives
     */
    public function testDetectsTwoDimensionalArrayOfPrimitives(array $array, bool $isPrimitives): void
    {
        $this->assertEquals($isPrimitives, ArrayUtility::isTwoDimensionalArrayOfPrimitives($array));
    }

    public function twoDimensionalArrayOfPrimitivesDataProvider(): array
    {
        return [
            [
                [ [ 'foo', 'bar', 'baz' ], [ 'second', 'line', 'here' ] ],
                true
            ],
            [
                [ 1, 2, 3 ],
                false
            ],
            [
                [ [ [ 'threeDimensional' ] ] ],
                false
            ],
            [
                [ 'twoDimensional' => [ 'but', 'associative' ] ],
                false
            ]
        ];
    }

    public function testEmptyExplodeIgnoresEmptyValues(): void
    {
        $this->assertEquals([], ArrayUtility::emptyExplode('#', ''));
        $this->assertEquals(['foo'], ArrayUtility::emptyExplode('#', 'foo'));
        $this->assertEquals(['foo'], ArrayUtility::emptyExplode('#', 'foo#'));
    }

}
