<?php

use \Newland\Toubiz\Api\Service\StringCleaner;

class StringCleanerTest extends \PHPUnit\Framework\TestCase
{
    public function testShouldStripWhiteSpace()
    {
        $this->assertEquals(
            '',
            (new StringCleaner())->cleanHtmlString('      ')
        );
    }

    public function testShouldStripHtmlEncodedWhiteSpace()
    {
        $this->assertEquals(
            '',
            (new StringCleaner())->cleanHtmlString('&nbsp;')
        );
    }

    public function testShouldNotStripPrintableCharacters()
    {
        $this->assertEquals(
            'test',
            (new StringCleaner())->cleanHtmlString('   test   ')
        );
    }

    public function testShouldKeepFormattedTags()
    {
        $this->assertEquals(
            '<p> </p><p> </p>',
            // Uses ascii character 194 instead of space character (32)
            (new StringCleaner())->cleanHtmlString('<p> </p><p> </p>')
        );
        $this->assertEquals(
            '<span> </span><span> </span>',
            // Uses ascii character 194 instead of space character (32)
            (new StringCleaner())->cleanHtmlString('<span> </span><span> </span>')
        );
        $this->assertEquals(
            '<i> </i><b> </b><u> </u>',
            // Uses ascii character 194 instead of space character (32)
            (new StringCleaner())->cleanHtmlString('<i> </i><b> </b><u> </u>')
        );
    }

    public function testShouldNotStripTagsWithPrintableCharactersInside()
    {
        $this->assertEquals(
            '<p>foobar</p><p> </p>',
            (new StringCleaner())->cleanHtmlString('<p>foobar</p><p> </p>')
        );
    }

    public function testShouldNotStripWhiteSpaceBetweenWords()
    {
        $this->assertEquals(
            'foo bar',
            (new StringCleaner())->cleanHtmlString('foo bar')
        );
    }


    public function testShouldNotStripWhiteSpaceBetweenTags()
    {
        $this->assertEquals(
            '<p>Foo</p> <span>bar</span>',
            (new StringCleaner())->cleanHtmlString('<p>Foo</p> <span>bar</span>')
        );
        $this->assertEquals(
            '<p>Foo </p><span> bar</span>',
            (new StringCleaner())->cleanHtmlString('<p>Foo </p><span> bar</span>')
        );
    }

    public function testShouldNotFailForStringsContainingAmpersand()
    {
        (new StringCleaner())->cleanHtmlString('<p>Foo & Bar</p>');
        (new StringCleaner())->cleanHtmlString(
            '<p><a href="https://www.postauto.ch/">Fahrplan Postauto</a></p>
<p>Passöffnung & Fahrplan der Postautos beachten</p>'
        );
        $this->assertTrue(true);
    }

    public function testShouldNotFailForStringsContainingBreaks()
    {
        (new StringCleaner())->cleanHtmlString('<br/>');
        (new StringCleaner())->cleanHtmlString('<br />');
        (new StringCleaner())->cleanHtmlString('<br>');
        (new StringCleaner())->cleanHtmlString('More text<br>Random Text');
        (new StringCleaner())->cleanHtmlString('<div></div><br>Random Text');
        (new StringCleaner())->cleanHtmlString('<div><br></div>');
        (new StringCleaner())->cleanHtmlString('<br></br>');
        (new StringCleaner())->cleanHtmlString('Random Text</br>');
        $this->assertTrue(true);
    }

    public function testShouldNotBreakOnNonTagAngleBracketsInText()
    {

        (new StringCleaner())->cleanHtmlString('Someone said >>oh no<< in incorrect french quotes');
        (new StringCleaner())->cleanHtmlString('Some stray < angle < brackets >');
        $this->assertTrue(true);
    }

    public function testReplacesDifferentKindsOfWhitespaceWithSingleSpace()
    {
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\nBar"), 'Should replace newline');
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\tBar"), 'Should replace tab');
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\rBar"), 'Should replace carriage return');
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\fBar"), 'Should replace formfeed');
        $this->assertEquals(
            'Foo Bar',
            StringCleaner::asString("Foo\vBar"),
            'Should replace vertical whitespace character'
        );
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xC2\xA0Bar", 'Should replace non breaking space'));
        $this->assertEquals(
            'Foo Bar',
            StringCleaner::asString("Foo\xC2\x82Bar", 'Should replace undefined control char')
        );
        $this->assertEquals(
            'Foo Bar',
            StringCleaner::asString("Foo\xC2\x86Bar", 'Should replace start of selected area')
        );
        $this->assertEquals(
            'Foo Bar',
            StringCleaner::asString("Foo\xC2\x8BBar", 'Should replace partial line forward')
        );
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xC2\x91Bar", 'Should replace private use only'));
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xC2\x97Bar", 'Should replace end guarded area'));
        $this->assertEquals(
            'Foo Bar',
            StringCleaner::asString("Foo\xC2\x9DBar", 'Should replace operating system command')
        );
    }

    public function testRemovesIncompleteUtf8Sequences()
    {
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xC2Bar"), 'Should remove \xC2');
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xADBar"), 'Should remove \xAD');
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xE2\x80Bar"), 'Should remove \xE2\x80');
    }

    /**
     * @dataProvider provideWellKnownSpecialCharacters
     * @param string $characters
     */
    public function testLeavesWellKnownSpecialCharacters(string $characters)
    {
        $this->assertEquals($characters, StringCleaner::asString($characters));
    }

    public function provideWellKnownSpecialCharacters(): array
    {
        return [
            'german umlauts' => [ 'äöüß' ],
            'french accents' => [ 'àâæç' ],
            'german quotes' => [ '„Verrückung”' ],
            'german quotes in hex' => [ "\u{201e}Verrückung\u{201d}" ],
            'currency symbols' => [ '€$£' ],
            'em-size dash' => [ '(1770–1843)' ],
        ];
    }

    public function testShouldRemoveControlCharacters()
    {
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\x10Bar"));
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\x15Bar"));
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\x1DBar"));
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xC2\x80Bar"));
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xC2\x8EBar"));
    }

    public function testShouldRemoveConfusingCharactersWithSpace()
    {
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xC2\xADBar"), 'Should remove soft hyphen');
        $this->assertEquals('Foo Bar', StringCleaner::asString("Foo\xE2\x80\xA8Bar"), 'Should remove line separator');
        $this->assertEquals(
            'Foo Bar',
            StringCleaner::asString("Foo\xE2\x80\xA9Bar"),
            'Should remove paragraph separator'
        );
        $this->assertEquals(
            'Foo Bar',
            StringCleaner::asString("Foo\xE2\x81\xA3Bar"),
            'Should remove invisible separator'
        );
    }

    public function testShouldRemoveNbspAsNumericValue()
    {

        $stringCleaner = new StringCleaner();

        $expected = 'NEU: am Mittag ein Tagesmenü für nur € 8,90 (Mo, Mi, Do, Fr) immer inklusive kleinem Salat.';
        $purified = $stringCleaner->purifyHtml(
            "<div>NEU:&#160;am&#160;Mittag ein Tagesmenü für\r\n    nur € 8,90&#160; (Mo, Mi, Do, Fr) immer inklusive kleinem Salat.</div>"
        );

        $this->assertEquals($expected, $stringCleaner->cleanWhiteSpace($purified));
    }

    /** @dataProvider provideStringsWithSpecialCharacters */
    public function testDoesNotProduceInvalidCodePointsWhenCleaningHtml(string $input): void
    {
        $this->expectNotToPerformAssertions();
        // Easy way of checking for UTF-8 validadity: try to json_encode it using safephp.
        // It'll throw an exception if the string is invalid
        Safe\json_encode((new StringCleaner)->cleanHtmlString($input));
    }

    public function provideStringsWithSpecialCharacters(): array
    {
        return [
            [ '<p>B500 Titisee- Feldberg Abzweig Lenzkirch - L4990 Lenzkirch</p> <p>47°52‘172‘‘N     8°12‘183‘‘O</p>' ]
        ];
    }
}
