<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Pool;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\Promise\RejectionException;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Exception\InvalidJsonException;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\EventApiService\EventAdapter;

/**
 * Service for legacy Toubiz event API.
 *
 * Concrete implementation to communicate with
 * the first, "old" toubiz API returning events.
 */
class EventApiService extends AbstractService
{
    use LanguageAware;

    const DEFAULT_BASE_URI = 'https://api.toubiz.de/json/event/';
    const ENDPOINT_LIST = 'json_result.php';
    const ENDPOINT_DETAIL = 'json_detail.php';
    const ENDPOINT_DELETED = 'ext/deleted.php';

    /** @var string[] */
    private $detailUriTemplates = [];

    public function setDetailUriTemplates(array $detailUriTemplates): void
    {
        $this->detailUriTemplates = $detailUriTemplates;
    }

    /**
     * Map of language constants to URI segments expected by the API.
     *
     * @var array
     */
    private static $languageMap = [
        Language::DE => 'de',
        Language::EN => 'en',
        Language::FR => 'fr',
        Language::ES => 'es',
    ];

    /**
     * @var string Legacy-specific organizer filter.
     */
    protected $organizerId;

    /**
     * Sets the organizer restriction.
     *
     * @api
     * @param string $organizerId
     * @return void
     */
    public function setOrganizerId($organizerId)
    {
        $this->organizerId = $organizerId;
    }

    public function fetchActiveEvents(\Closure $block, \Closure $onProgress): ServiceResult
    {
        $list = $this->listEvents($onProgress)->wait();
        $this->fetchEventsDetails($list, $block)->wait();

        $deleted = $this->fetchDeletedIds()->wait();

        $result = new ServiceResult();
        $result->setAll($list);
        $result->setDeleted($deleted);
        $result->setSuccess(true);

        return $result;
    }

    public function fetchEventsDetails(array $identifiers, \Closure $block = null): PromiseInterface
    {
        $requests = array_map(
            function (string $id) {
                return $this->prepareRequest(static::ENDPOINT_DETAIL, [ 'id' => $id ]);
            },
            $identifiers
        );

        $adapters = [];
        $total = \count($identifiers);
        $pool = new Pool(
            $this->httpClient,
            $requests,
            [
                'concurrency' => $this->parameters['concurrency'] ?? 10,
                'fulfilled' => function (Response $response) use ($block, $total, &$adapters) {
                    $data = $this->handleJsonResponse($response);
                    $adapter = $this->bodyToAdapter($data);
                    if ($adapter) {
                        $adapters[] = $adapter;
                        if ($block) {
                            $block($adapter, $total, 'Importing');
                        }
                    }
                },
            ]
        );

        return $pool->promise()->then(
            function () use (&$adapters) {
                return $adapters;
            }
        );
    }

    private function listEvents(callable $onProgress): PromiseInterface
    {
        $onProgress(0, -1, 'Listing events');
        $pages = -1;
        $itemsPerPage = 100;
        $i = 0;

        $pool = new ConcurrentPaginatedRequests(
            $this->parameters['concurrency'] ?? 10,
            function (int $page) use (&$i, &$pages, $itemsPerPage, $onProgress) {
                return $this->requestEventList($page, $itemsPerPage)->then(
                    function (array $body) use (&$pages, &$i, $onProgress) {
                        $pages = $body['pages'];
                        if (empty($body['result'])) {
                            throw new RejectionException('End of pages reached');
                        }
                        $onProgress(++$i, $pages, 'Listing events');
                        return $body;
                    }
                );
            }
        );


        return $pool->start()->then(\Closure::fromCallable([ $this, 'extractEventIdsFromResponseBodies' ]));
    }

    private function requestEventList(int $page, int $itemsPerPage): PromiseInterface
    {
        return $this->sendRequest(
            static::ENDPOINT_LIST,
            [ 'offset' => $itemsPerPage * ($page - 1), 'per_page' => $itemsPerPage ]
        );
    }

    /**
     * Extracts the event ids from an array of parsed responses from the list api.
     *
     * @param array $responses
     * @return string[]
     */
    private function extractEventIdsFromResponseBodies(array $responses): array
    {
        $now = (new \DateTime())->getTimestamp();
        $events = [];

        foreach ($responses as $response) {
            foreach ($response['result'] as $result) {
                $modified = new \DateTime($result['timestamp']);
                if ($modified->add($this->delta)->getTimestamp() > $now) {
                    $events[] = $result['id_event'];
                }
            }
        }

        return $events;
    }

    private function fetchDeletedIds(): PromiseInterface
    {
        $time = (new \DateTime())->sub($this->delta)->getTimestamp();
        $responseToIds = function (array $body) {
            return array_map(
                function (array $item) {
                    return $item['id_event'];
                },
                $body['result']
            );
        };

        return $this->sendRequest(static::ENDPOINT_DELETED, [ 'timestamp' => $time ])->then($responseToIds);
    }

    protected function sendRequest(string $path, array $query): PromiseInterface
    {
        $request = $this->prepareRequest($path, $query);
        return $this->httpClient->sendAsync($request)->then(
            function (Response $response) {
                return $this->handleJsonResponse($response);
            }
        );
    }

    protected function handleJsonResponse(Response $response): ?array
    {
        if ($response->getStatusCode() !== 200) {
            return null;
        }

        $dataRaw = $response->getBody();
        $dataRaw = StringCleaner::asString($dataRaw);
        $data = json_decode($dataRaw, true);

        if ($data === null) {
            throw new InvalidJsonException(
                'API request returned invalid JSON.',
                1557149751
            );
        }

        return $data;
    }

    protected function bodyToAdapter(?array $body): ?EventAdapter
    {
        if ($body === null) {
            return null;
        }

        $adapter = new EventAdapter($body);
        $adapter->setDetailUriTemplates($this->detailUriTemplates);

        if ($this->language) {
            $adapter->setLanguage($this->language);
        }

        return $adapter;
    }

    protected function prepareRequest(string $path, array $parameters = []): Request
    {
        $parameters = array_merge(
            $parameters,
            [
                'key' => $this->apiKey,
                'mandant' => $this->clientName,
                'organizer' => $this->organizerId,
                'language' => static::$languageMap[$this->language] ?? 'de',
            ]
        );

        $uri = Uri::withQueryValues(new Uri($path), $parameters);
        return new Request('GET', $uri);
    }
}
