<?php declare(strict_types=1);

namespace Newland\GpsFileParsing\Model;

class Point implements \JsonSerializable
{
    protected const EARTH_RADIUS_KILOMETERS = 6371;

    /** @var float */
    protected $latitude;

    /** @var float */
    protected $longitude;

    /** @var float|null */
    protected $elevation;

    public function __construct(float $latitude, float $longitude, float $elevation = null)
    {
        $this->latitude = $latitude;
        $this->longitude = $longitude;
        $this->elevation = $elevation;
    }

    public function getLatitude(): float
    {
        return $this->latitude;
    }

    public function getLongitude(): float
    {
        return $this->longitude;
    }

    public function getElevation(): ?float
    {
        return $this->elevation;
    }

    public function isHigherThan(Point $point): bool
    {
        return $this->elevation > $point->getElevation();
    }

    public function isLowerThan(Point $point): bool
    {
        return $this->elevation < $point->getElevation();
    }

    /**
     * Returns the distance from this points to another point in kilometers.
     * The Haversine formula allows to calculate the distance between two points
     * TODO: Take into account elevation with the Pythagorean theorem for more precise distances
     *
     * @param Point $pointTo
     * @return float $distance
     */
    public function distance(Point $pointTo): float
    {
        // As we are working on a sphere, we need radians
        // Therefore we convert our latitudes and longitudes
        $radLatitudeFrom = deg2rad($this->latitude);
        $radLongitudeFrom = deg2rad($this->longitude);
        $radLatitudeTo = deg2rad($pointTo->latitude);
        $radLongitudeTo = deg2rad($pointTo->longitude);

        // Then we need the delta between the latitudes and longitudes
        $latitudeDelta = $radLatitudeTo - $radLatitudeFrom;
        $longitudeDelta = $radLongitudeTo - $radLongitudeFrom;

        // Apply the Haversine formula with the different variables
        $angle = 2 * asin(
            sqrt(
                (sin($latitudeDelta / 2) ** 2) +
                cos($radLatitudeFrom) *
                cos($radLatitudeTo) *
                (sin($longitudeDelta / 2) ** 2)
            )
        );

        // Return the result, formatted in the wished unit of measurement
        return $angle * static::EARTH_RADIUS_KILOMETERS;
    }

    public function jsonSerialize()
    {
        return [
            $this->latitude,
            $this->longitude,
        ];
    }
}
