<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Domain\Model;

use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Tests\Factory\CategoryFactory;

class CategoryTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /**
     * @dataProvider provideDeterministicIds
     * @param string $expectedId
     * @param array $properties
     */
    public function testDeterministicIdsAreGeneratedCorrectly(string $expectedId, array $properties): void
    {
        $this->assertEquals(
            $expectedId,
            (new CategoryFactory($this->objectManager))->make($properties)->generateUuid()->toString(),
            'persistence object identifier should be stable: If this test fails then the'
            . ' way ids are being generated has probably changed and old identifiers need to be migrated'
            . ' to this new style.'
        );
    }

    /**
     * @dataProvider provideDeterministicIds
     * @param string $expectedId
     * @param array $properties
     */
    public function testDeterministicIdsAreAutomaticallyGeneratedWhenStoringInDatabase(string $expectedId, array $properties): void
    {
        /** @var Category $category */
        $category = (new CategoryFactory($this->objectManager))->create($properties);

        $this->assertEquals(
            $expectedId,
            $category->getPersistenceObjectIdentifier(),
            'persistence object identifier should be stable: If this test fails then the'
            . ' way ids are being generated has probably changed and old identifiers need to be migrated'
            . ' to this new style.'
        );

        // Entity is explicitly deleted to avoid id collisions.
        $this->persistenceManager->remove($category);
        $this->persistenceManager->persistAll();
    }

    public function provideDeterministicIds(): array
    {
        return [
            'baseCategory' => [
                'd8dfa07e-568d-575b-888a-277a603eaa7a',
                [
                    'language' => 'de',
                    'originalId' => 'lalala',
                    'title' => 'First name'
                ]
            ],
            'sameAsBaseButDifferentName: Should have same identifier' => [
                'd8dfa07e-568d-575b-888a-277a603eaa7a',
                [
                    'language' => 'de',
                    'originalId' => 'lalala',
                    'title' => 'Same basic category as before but with different title.'
                ]
            ],
            'languageChanged' => [
                '12530010-5515-5d6f-9b1f-cf98a4b53d6e',
                [
                    'language' => 'en',
                    'originalId' => 'lalala',
                ]
            ],
            'originalIdChanged' => [
                '6e468d94-973c-5b7b-bad9-fd8a36a1375d',
                [
                    'language' => 'de',
                    'originalId' => 'blablabla',
                ]
            ],
            'differentPropertiesAllTogether' => [
                '5b71f71f-459c-5bb3-afad-5adc9ff42ac1',
                [
                    'language' => 'fr',
                    'originalId' => 'foooooo',
                ]
            ],
        ];
    }

}