<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\LinkHandler;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Newland\NeosCommon\Domain\Repository\NodeRepository;
use Newland\NeosCommon\LinkHandler\Domain\Model\Record;
use Newland\NeosCommon\LinkHandler\Handler\AbstractLinkHandler;
use Newland\NeosCommon\LinkHandler\LinkingService;
use Newland\NeosCommon\Service\NodeService;
use Neos\Flow\Annotations as Flow;

class FilterItemLinkHandler extends AbstractLinkHandler
{

    /**
     * @var NodeRepository
     * @Flow\Inject()
     */
    protected $nodeRepository;

    /**
     * @var NodeService
     * @Flow\Inject()
     */
    protected $nodeService;

    /**
     * @var LinkingService
     * @Flow\Inject()
     */
    protected $linkingService;

    /**
     * @inheritDoc
     */
    public function generateRecordLink(
        Record $record,
        ControllerContext $controllerContext,
        NodeInterface $contextNode,
        bool $absolute
    ): ?string {
        $node = $this->findNode($record->getId());
        if (!$node) {
            return null;
        }

        $documentNode = $this->nodeService->getDocumentNode($node);
        if (!$documentNode) {
            return null;
        }

        return sprintf(
            '%s#%s',
            $this->linkingService->createNodeUri($controllerContext, $documentNode, $documentNode, 'html', $absolute),
            $record->getId()
        );
    }

    /**
     * @inheritDoc
     */
    public function findByIdentifier(string $recordIdentifier): ?Record
    {
        $node = $this->findNode($recordIdentifier);
        if (!$node) {
            return null;
        }

        $record = new Record();
        $record->setId($node->getIdentifier());
        $record->setTitle($node->getProperty('title'));
        return $record;
    }

    private function findNode(string $identifier): ?NodeInterface
    {
        $query = $this->nodeRepository->queryForNodeType('Newland.Toubiz.Map.Neos:Map.FilterItem', 'node');
        $query->andWhere($query->expr()->eq('node.identifier', ':identifier'));
        $query->setParameter('identifier', $identifier);
        $query->setMaxResults(1);
        return $this->nodeRepository->executeNodeDataQuery($query)[0] ?? null;
    }

    /**
     * @inheritDoc
     */
    public function findBySearchTerm(string $searchTerm): array
    {
        $searchTerm = strtolower($searchTerm);

        // Query for properties as a first stage so we don't grab all data from the db
        $query = $this->nodeRepository->queryForNodeType('Newland.Toubiz.Map.Neos:Map.FilterItem', 'node');
        $query->andWhere($query->expr()->like('node.properties', ':query'));
        $query->setParameter('query', sprintf('%%%s%%', $searchTerm));


        $result = [];
        foreach ($this->nodeRepository->executeNodeDataQuery($query) as $node) {
            $title = strtolower($node->getProperty('title') ?: '');
            if (strpos($title, $searchTerm) !== false) {
                $record = new Record();
                $record->setId($node->getIdentifier());
                $record->setTitle(sprintf(
                    '%s [%s]',
                    $node->getProperty('title'),
                    'Kartenmenü'
                ));
                $result[] = $record;
            }
        }

        return $result;
    }
}
